#Loading the necessary libraries
using Plots
using ModelingToolkit
using DifferentialEquations
using Random
using Statistics
using OrdinaryDiffEq
using Lux 
using DiffEqFlux
using Flux
using ComponentArrays 
using Optimization, OptimizationOptimJL, OptimizationOptimisers                                                                   
rng = Random.default_rng()
Random.seed!(99)

#Constants
C = 0.01


#Initial Conditions
I = [1, 0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)

#Define the whitedwarf equation as a function
function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)

#Plot
plot(sol, linewidth = 1, title = "White Dwarf equation", xaxis = "\\eta",
     label = ["\\phi" "\\phi'"])

#--------------I will solve the white dwarf equation using the SecondOrderODEProblem function------------

#Defining the function containing the Second Order Differential Equation
function whitedwarf2(ddu,du,u,C,eta)
    ddu .= (-((u.*u.-C)).^(3/2) - 2/eta * du)
end

#Initial conditions definined as required by the syntax of the Second Order Differential Equation
dpsi0=[0.0]
psi0=[1.0]
#Defining the secondOrderProblem 
prob2 = SecondOrderODEProblem(whitedwarf2,dpsi0, psi0, etaspan, C)
#Solving it with the algorithm
sol2 = solve(prob2, saveat=etasteps)

#plot sol2
plot(sol2, linewidth=1.5, title = "White Dwarf equation", xaxis = "\\eta", label = ["\\phi" "\\phi '"])




#Adding moderate noise to data:

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 7e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data
plot(sol, alpha = 0.75, color = :black, label = ["True Data" nothing])
scatter!(sol.t, transpose(x1_noise), color = :red, label = ["Noisy Data" nothing])


#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)


#Selecting the portion of the training data out of the full data
etasteps =  etasteps[1:end-80]
etaspan = (etasteps[1], etasteps[end])


prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-80]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end


callback = function (p, l, pred; doplot = true)
    println(l)
    # plot current prediction against data
    if doplot

        plt1 = scatter(collect(etasteps), true_data[1, :]; label = "\\phi data")
        scatter!(plt1, collect(etasteps), pred[1, :],markershape=:xcross; label = "\\phi prediction")
        scatter!(plt1, collect(etasteps), true_data[2, :]; label = "\\phi ' data")
        scatter!(plt1, collect(etasteps), pred[2, :],markershape=:xcross; label = "\\phi ' prediction")
        #plt1 = scatter(sol.t, true_data[3, :]; label = "data")
        #scatter!(plt1, sol.t, pred[3, :]; label = "prediction")
        #plt=plot(plt1, plt2)
        
        display(plot(plt1))

        
        
    end
    return false
end


pinit = ComponentArray(p)
callback(pinit, loss_neuralode(pinit)...; doplot = true)




# use Optimization.jl to solve the problem
adtype = Optimization.AutoZygote()

optf = Optimization.OptimizationFunction((x, p) -> loss_neuralode(x), adtype)
optprob = Optimization.OptimizationProblem(optf, pinit)

result_neuralode = Optimization.solve(optprob, OptimizationOptimisers.Adam(0.05); callback = callback,
    maxiters = 150)

optprob2 = remake(optprob; u0 = result_neuralode.u)

result_neuralode2 = Optimization.solve(optprob2, Optim.BFGS(; initial_stepnorm = 0.01);
    callback, allow_f_increases = false, maxiters=100)



p_trained = (layer_1 = (weight = Float32[-0.54207605 -0.08931858; -0.59132355 -0.044312436; 1.0597245 -0.2904367; -0.8415495 0.2518355; -0.9635182 0.34537458; 0.7465145 -0.16712336; 0.08815579 0.0463996; -0.42038482 0.43479463; -0.47389007 0.038572628; -1.0732368 0.3044327; 1.0243837 -0.24159656; 1.3783909 -0.36483982; -0.90717334 0.049350295; 0.24813363 -0.10815106; 0.7474638 -0.09410347; -0.39710385 -0.036752835; -0.87189215 0.2045615; 0.8214552 -0.39658958; -1.299256 0.087683894; 0.26793545 -0.12529227; 1.1832998 -0.33983886; -0.94385207 0.15146747; -0.329743 0.06348516; -0.43338436 -0.017511755; -1.2315717 0.30009922; -1.2831876 0.339342; -0.5267163 -0.079912215; 0.5056784 -0.31040782; -1.4039207 0.3399155; -0.75088274 -0.013748293; -0.20255332 -0.27451023; -0.019680737 -0.11039811; -1.3249797 0.48787087; 1.2282455 -0.42017147; -0.5807599 0.26207042; -0.2811132 -0.09922412; -0.9019461 0.17038432; -0.41125125 0.0017450065; -0.4238288 0.26609135; 1.0131577 -0.3712033; 0.98539764 -0.55725086; 0.8064437 -0.20428264; -1.5322368 0.43054503; 0.32411274 -0.07060059; -0.41011345 0.4305979; 0.56363606 -0.24253629; -0.38346055 0.3951012; 0.10675447 0.010848249; -0.31692457 -0.026318805; 0.47868532 -0.42820114; 0.41431135 0.012447005; -0.14703518 0.38260275; 0.6048018 -0.23120809; 0.34514105 -0.07317293; -0.8864168 0.27359483; -1.4898992 0.37373707; -0.006542587 0.20440874; 0.5671207 -0.06769265; 0.28469294 0.08822053; -0.3840529 0.14991032; 0.49310872 -0.041024636; 1.0778338 -0.2636088; 0.6775203 -0.0873828; -0.6997273 0.058863312; -0.1495981 0.04757312; 0.61004806 -0.03194547; -1.1729559 0.27103978; -0.82104003 0.20105867; 0.16132554 0.14542112; 1.6226861 -0.4160599; 0.75758433 -0.35831085; 1.1325083 -0.402214; -0.05889494 -0.21251523; -0.006180583 -0.15856011; -0.6403392 0.05619576; 0.22551152 0.11672373; 1.2285992 -0.3345474; -1.0379939 0.3745136; -0.34630528 -0.034626774; 1.5791305 -0.5047022], bias = Float32[0.37308487; 0.21701902; -1.2334192; 0.8260774; 0.9162292; -0.86153036; 0.05204648; 0.4532263; 0.46903986; 1.3075551; -0.8268877; -1.2375834; 0.4837869; -0.47050542; -0.55511963; 0.3067771; 0.77325886; -1.3715938; 0.735002; -0.42872128; -1.2354984; 0.7851556; 0.4700865; 0.07994914; 0.8489315; 1.1154802; 0.7277209; -0.256047; 1.5120263; 0.5170015; -0.077014506; 0.31515408; 1.2225983; -1.5816892; 0.85873365; 0.09612368; 0.7107153; 0.23812811; 0.32308543; -1.5175412; -1.1841085; -0.7375026; 1.2855228; -0.109926336; 0.7115491; -0.47920826; 0.39693955; -0.113968685; 0.16485624; -0.17176494; 0.12407618; 0.23504265; -0.7331125; -0.29360488; 1.1643862; 1.537784; 0.102274075; -1.2043313; 0.034792528; 0.6482183; -0.36256468; -1.111788; -1.12302; 0.28643304; 0.6912492; -0.21754004; 0.84342736; 0.71072686; -0.29240483; -1.2076862; -1.357724; -1.4453431; 0.6272033; -0.13705032; 0.4241586; -0.11141496; -1.5861309; 1.1541259; 0.2740176; -1.5909904]), layer_2 = (weight = Float32[0.33461708 0.40879425 -0.36993825 0.1701881 0.24477766 0.15559928 -0.22776048 0.04796663 0.0070090927 0.3818212 -0.13073488 -0.41275206 0.39544797 -0.039934143 -0.018612362 0.035455965 0.2580012 -0.242024 0.41469488 -0.09398249 -0.06028988 0.28465375 -0.11953456 0.0026025174 0.23301445 0.2870219 0.25547594 -0.23258425 0.2802397 0.3175477 -0.08895209 -0.027606698 0.45601678 -0.2742371 0.1903645 -0.09679374 -0.052920043 -0.27070498 0.033933416 -0.55571955 -0.4096789 -0.24693134 0.23060699 -0.1144066 0.032127492 -0.54904073 0.058329757 0.0060819816 -0.05770674 -0.3387165 -0.2967032 0.2505008 -0.30382854 -0.1356858 0.41809723 0.3376441 -0.09442626 -0.42520228 0.11439814 -0.022213994 -0.13559046 -0.48232228 -0.57054126 -0.09537159 -0.23761447 -0.05018299 0.1827251 -0.0667877 -0.21270967 -0.2115679 -0.34794614 -0.5310534 0.39442444 -0.15292922 -0.0048851646 -0.39370942 -0.36799452 0.2671857 -0.01971972 -0.2843546; 0.032274522 0.039591003 -0.3825252 0.27928075 0.17721282 -0.20794383 0.26482415 -0.3454737 0.19461551 0.39305493 -0.4563102 -0.33457145 0.2641643 -0.28616232 -0.27276167 0.49187604 0.48657298 -0.29366168 0.2285911 -0.46896392 -0.22322214 0.22012793 0.35467285 0.35845703 0.16574182 0.2735118 0.116315216 0.27316004 0.4749284 0.113258116 0.2921594 0.48114666 0.3470539 -0.33070964 0.32394665 0.29355732 0.49087638 0.32397652 0.28157052 -0.4050132 -0.24168347 -0.16814966 0.15511025 -0.18558836 0.31931323 -0.4202532 -0.29903382 0.4693086 0.36054444 0.2825776 0.17691112 0.3031081 -0.14213502 -0.29061437 -0.10131075 0.4008535 -0.18477783 0.04834919 -0.5603311 0.38998875 -0.4708652 -0.38876185 -0.38250098 0.45438465 -0.32824492 -0.48416623 0.5157436 0.4725154 0.17956159 -0.50764316 -0.40125033 -0.3051873 -0.16301297 0.2562499 0.4493927 -0.27303767 -0.2561733 0.3754286 0.23946829 -0.44456998; -0.24148637 -0.16795222 0.4607801 -0.3355346 -0.25774363 -0.015176845 0.29012147 -0.21237698 -0.0065202946 -0.22457142 0.14438339 0.41379273 -0.45049483 -0.15663512 -0.17677458 0.08917911 -0.4176995 0.24675998 -0.38009098 -0.2106186 0.15787788 -0.38807192 0.16590247 0.21646032 -0.57077676 -0.29943234 -0.43128443 0.26389003 -0.5972838 -0.23376124 0.015098338 0.006419941 -0.33853567 0.27817732 -0.4588638 -0.025573444 -0.07255597 -0.005079637 0.03341348 0.42135328 0.58953303 0.36155555 -0.36269936 0.06045511 -0.06166567 0.16784884 -0.32560107 0.23073606 -0.02403361 0.02427311 0.08750923 0.10311062 0.42297834 -0.21036011 -0.514128 -0.5030127 0.19012967 0.36945683 -0.09834681 -0.07063966 0.09116411 0.50236654 0.4296619 -0.0720239 -0.19884235 -0.10007975 -0.22188927 0.022894526 0.09979626 0.31267905 0.49040562 0.28298935 -0.10001983 0.17176116 -0.045907646 0.24422684 0.48706794 -0.4724676 0.13374212 0.6249627; 0.29367644 0.23053057 -0.16998759 0.449879 0.25998586 0.027147468 -0.06409859 -0.21435606 -0.2885 0.29986647 -0.13045391 -0.269464 0.40568376 -0.20368138 -0.12005806 -0.10270766 0.42399237 -0.37929714 0.30645072 -0.12945868 -0.18000078 0.4576558 -0.15483838 -0.21659131 0.04809207 0.20906068 0.471115 0.22358292 0.16596626 0.46599752 -0.124349646 -0.22374265 0.2474959 -0.32385382 0.16963907 -0.24859536 0.08962067 -0.11042872 0.17022389 -0.37802488 -0.44325605 -0.20776054 0.0065459316 -0.13330181 0.23819885 -0.21192233 0.075472906 0.33784112 -0.02404806 -0.23951097 0.20146711 0.20426202 -0.35193875 -0.39350095 0.19323456 0.22377196 -0.16139601 -0.29938653 0.0031855016 0.289096 -0.12231396 -0.22405347 -0.27044958 0.064689785 -0.28061432 0.14357212 -0.06623207 -0.1954775 -0.2122382 -0.30719832 -0.28975236 -0.071217366 0.18839268 0.10061283 -0.111003526 -0.38097683 0.093410775 0.24471965 0.022553625 -0.1955609; -0.048153035 0.031856976 0.23906457 -0.18929759 -0.009841381 -0.14396904 0.09813737 -0.32673827 0.13079235 -0.22376078 -0.04945383 0.1870933 -0.14060329 -0.20257291 -0.20394015 0.01971409 -0.14305183 0.4313679 -0.43218523 -0.26884028 0.30711457 -0.23704958 0.20714147 0.31437272 -0.5189917 -0.1819319 -0.15191823 0.24563493 -0.46264037 -0.28504542 0.16505383 0.24080339 -0.454224 0.7643068 0.035883367 0.31757653 0.048025828 0.34799913 0.03691814 0.46291846 0.36428943 -0.1462214 -0.591154 -0.19150732 0.2648118 0.0667382 -0.213536 0.37067753 0.1952312 0.2313561 0.3716392 0.10906565 0.27728727 -0.16933481 -0.4263679 -0.3652412 0.1103169 0.5203278 -0.11125016 0.18689626 -0.28156456 0.41764554 0.29827964 0.2330748 -0.4546563 -0.17693469 0.17369851 -0.12271715 0.23839259 0.31550923 0.40619695 0.5892376 -0.15148096 0.41509265 0.212732 0.016639419 0.21572426 -0.32489184 0.15077642 0.62432086; -0.036115035 -0.28458804 0.18495344 -0.18259765 -0.41078892 0.27461177 0.1294712 -0.064605184 0.15784022 -0.24422519 0.2647911 0.5151844 -0.48436943 0.120461755 0.20497957 0.072211914 -0.22240545 0.36903492 -0.15347579 -0.0018583358 0.4339104 -0.36735848 -0.17982626 0.09635181 -0.2361838 -0.5007165 -0.4244071 -0.08813404 -0.123172104 -0.10110672 0.06386467 0.09609763 -0.27519226 0.4253258 -0.46667755 0.19613594 -0.28774792 -0.11607527 -0.25617298 0.54573697 0.3635491 0.32812214 -0.45266822 0.11839287 -0.18931414 0.13599169 0.008363336 -0.19634897 0.11337961 0.33459473 -0.12172101 -0.3786094 0.22521809 0.21925016 -0.25822926 -0.14821334 -0.05335884 0.3388616 -0.22289643 -0.08963179 -0.053515553 0.2617658 0.3531535 -0.009884014 0.08596881 0.09309446 -0.048428055 0.09593992 0.1457005 0.23171681 0.18996385 0.15616249 -0.038278624 -0.15641052 -0.10496756 0.25343552 0.42899397 -0.4049709 -0.12824467 0.36801884; 0.24054505 0.089625366 -0.36320183 0.17057191 0.15906486 -0.31875834 0.0043747406 0.246997 0.13685212 0.47875452 -0.15312818 -0.56045586 0.22338563 -0.09935357 -0.19692671 -0.174563 0.15874347 -0.42736745 0.43772328 -0.096111305 -0.29488224 0.5211183 0.04525216 0.015687091 0.45270422 0.56541365 0.15075633 -0.08586343 0.5045801 0.23019399 -0.2523985 -0.10070981 0.28050604 -0.46600166 0.16684626 -0.103705615 0.3649541 -0.038764622 0.08405514 -0.25303492 -0.21081427 -0.3202844 0.56165075 -0.07883257 0.22994626 -0.38212985 0.10124075 -0.13623242 0.07338872 -0.2130802 -0.12674543 0.2608629 -0.22332297 -0.25587195 0.19817738 0.3080907 -0.056165032 -0.13495609 0.04564324 0.22969534 -0.22877108 -0.19964448 -0.4442337 0.02564152 -0.06321247 -0.12355311 0.21893975 0.33525553 -0.018024776 -0.66491616 -0.5466549 -0.30440906 0.17538033 -0.23632838 0.12979645 0.023962129 -0.5493937 0.393527 0.11415945 -0.61616576; 0.28477442 0.13231048 -0.3956383 0.29920095 0.24095291 -0.0053778533 -0.2717027 0.2062244 -0.2530767 0.18515939 -0.23191406 -0.54560477 0.4092828 0.049972393 -0.099284746 0.019556027 0.189197 -0.18764631 0.35769144 0.14685367 -0.06699311 0.32437938 0.024041642 0.015870722 0.30605057 0.29987842 0.4196245 -0.22519405 0.14221707 0.30173573 -0.35900542 -0.021812467 0.6299222 -0.467631 0.24050453 -0.027793022 0.17042221 -0.34555176 0.06061227 -0.4505182 -0.17415315 -0.38568446 0.43168935 -0.056520462 0.05164402 -0.1705001 0.029336868 -0.030244105 0.0032610302 -0.36357874 -0.28693333 -0.061574824 -0.14081836 0.15872909 0.44020572 0.3139284 -0.017886255 -0.42708677 0.08619214 0.108633436 -0.16077031 -0.46032256 -0.47868434 -0.14795114 -0.06706061 -0.04245086 -0.022358483 -0.049947858 -0.2345475 -0.3061241 -0.36968026 -0.19682916 0.15550607 0.003087511 0.052613515 -0.35091317 -0.20617855 0.19240521 0.06206283 -0.3821095; 0.028890861 -0.03625557 0.26114687 -0.19588642 -0.42114615 0.528907 -0.40293035 0.20066181 -0.45452043 -0.3107287 0.41076458 0.124909066 -0.13512415 0.18910788 0.18480092 -0.29384732 -0.17227514 0.28472883 0.04404032 0.23840117 0.19666867 0.29694325 -0.35446435 -0.40191802 -0.25396177 -0.28458744 0.21606532 -0.09105273 -0.19813268 0.15131901 -0.42611763 -0.5772237 -0.40524325 0.18303911 -0.13966143 -0.4917223 -0.21415348 -0.3096811 -0.207697 0.28661543 0.27021697 0.4718522 -0.23594448 0.21763723 -0.489714 -0.020477582 0.403677 -0.18867226 -0.3421048 -0.3534773 -0.28295523 -0.25198683 0.30347544 0.23727185 0.29695818 -0.46170145 -0.17778769 -0.49505183 0.45972517 -0.42308754 0.25512633 0.2444528 -0.016555674 -0.2779863 0.14073275 0.48444813 -0.18850185 -0.12754737 -0.0151145635 0.10410626 0.34977496 0.45141953 0.20091793 -0.36889908 -0.50660837 0.1598793 0.3733977 -0.488712 -0.37195763 0.29360268; 0.06779519 0.010259952 0.51378644 -0.32839262 -0.34638172 0.50935364 -0.27158394 -0.15871195 -0.1947988 -0.3373726 0.44402075 0.28781423 0.01686998 0.24381931 0.15486814 -0.3481941 -0.30140263 0.35227415 0.08073248 0.22581106 0.22929576 -0.1547024 -0.3896421 -0.43720186 -0.05464422 -0.575493 0.12886861 0.0026862926 -0.47886765 -0.26081184 0.0043796026 -0.4955135 -0.5028993 0.29304925 -0.1443823 -0.30581957 -0.27418706 -0.30177054 -0.2850587 0.3341473 0.6008719 0.28481978 -0.2830336 0.028848266 -0.4466957 0.15819141 -0.08910892 -0.07499038 -0.30239797 -0.12820947 -0.2761379 -0.053882003 0.09971828 0.2493402 -0.0097504575 -0.47383142 -0.08320839 0.085884154 0.49123225 -0.42024532 0.5263166 0.14726204 0.058854092 -0.3402593 0.06171838 0.37499595 -0.36735728 -0.29756975 0.0005468922 0.4832171 0.33178902 0.21414834 0.14311492 -0.17668724 -0.3403998 -0.04195328 0.5726623 -0.5781713 -0.43364418 0.2730859; 0.12880786 0.29522392 -0.2562987 0.36733446 0.10904668 -0.13839372 -0.055444695 0.049798146 0.1425636 0.15426101 -0.23080951 -0.25790465 0.5886965 -0.18018913 -0.18747245 0.1439605 0.460486 -0.22987463 0.30448425 -0.11996192 -0.28481728 0.5097976 -0.009006133 0.06979278 0.41374445 0.19726102 0.18186824 -0.26418966 0.31213427 0.24426271 -0.033711698 -0.008122582 0.6018059 -0.23673877 0.21064803 0.021396268 0.083463676 -0.06166815 0.07463428 -0.32707563 -0.39921182 -0.3914349 0.56845975 -0.23213725 0.112780355 -0.20409665 -0.059155587 -0.035408754 -0.012874074 -0.113444686 -0.14690697 0.18995102 -0.27834332 -0.3031972 0.39772034 0.4537729 -0.17259362 -0.34140766 0.13300283 0.13244115 -0.21869725 -0.2717875 -0.28779215 0.2348397 -0.06396554 -0.20766982 0.24355794 0.16958217 -0.25444013 -0.59180737 -0.3181682 -0.18909602 0.26103908 -0.09965679 -0.10413425 -0.24112277 -0.33556944 0.32354343 0.08109841 -0.22953542; 0.42063025 0.21498756 -0.4899378 0.28346068 0.23904143 -0.34126744 -0.001540922 0.035433453 0.08841906 0.17410369 -0.31592315 -0.36495128 0.34520248 -0.11128815 -0.40070662 0.026311714 0.4506319 -0.2632558 0.41229713 -0.1919225 -0.3154817 0.27992252 0.17983176 -0.028810434 0.3011493 0.16331562 0.4804012 -0.25742802 0.31248805 0.42269364 -0.051383138 0.042432215 0.32749802 -0.48378497 0.34356728 0.06490993 0.16158874 0.03016832 0.06640869 -0.48262283 -0.25811806 -0.4040287 0.34523883 -0.17370006 0.22163448 -0.23459935 0.006687311 0.32997644 0.097122796 -0.21449757 -0.13696077 0.09967865 -0.1771323 -0.31733224 0.3929034 0.31913313 -0.37148502 -0.32791105 0.06443982 0.07748938 -0.2168896 -0.37135592 -0.60585815 0.09813786 -0.017855486 -0.30616924 0.17766748 0.30361938 -0.09502518 -0.304009 -0.1740269 -0.4104346 0.15542781 -0.05959146 0.23305105 -0.40818188 -0.47776031 0.39442056 0.21047649 -0.35807607; 0.3517906 0.25669894 0.24009992 -0.2933264 -0.21832982 0.3843834 -0.5064912 0.22662121 -0.19454575 -0.4579296 0.2608241 -0.15910628 0.100053206 0.44869995 0.29631284 -0.4340043 -0.24020028 0.17609736 0.24758951 0.4445661 0.41913548 0.3379902 -0.26842496 -0.25154746 -0.042362425 -0.13913523 0.16452302 -0.12966998 -0.25607026 0.0020253055 -0.1836872 -0.45134377 0.07798874 0.39965278 -0.4141949 -0.43168882 -0.3819784 -0.32980397 -0.4891655 -0.09420176 0.13506591 0.47339457 0.0594885 0.21121407 -0.2907123 0.09145878 0.3771282 -0.43758652 -0.55738336 -0.3838497 -0.332151 -0.49011207 0.02113285 0.2545777 0.4116036 -0.40879798 0.06909613 -0.27261156 0.38555276 -0.42875293 0.5133954 0.08548068 -0.20132113 -0.3043841 0.118788876 0.46820524 -0.26138166 -0.5199695 -0.14607224 0.18077543 0.3493828 0.22491693 0.45312247 -0.17802423 -0.43988556 -0.09023301 0.43409747 -0.31808645 -0.23620602 0.08794565; -0.2769421 -0.16674009 0.2901834 -0.1432678 -0.473246 0.4581039 -0.11552601 -0.17641832 -0.18066174 -0.25777674 0.42257336 0.57632184 -0.4378541 0.17835967 0.36631238 -0.25849903 -0.33915383 0.5555749 -0.44479048 0.039486825 0.5590794 -0.19098108 -0.25522766 -0.0007131875 -0.23174982 -0.550227 -0.096336946 0.33953923 -0.61303276 -0.26514158 0.062400557 0.16017672 -0.3731647 0.62764984 -0.31297928 0.13681635 -0.2901096 -0.121208884 -0.36211738 0.510611 0.5464224 0.40652588 -0.5931786 0.024592936 -0.22046652 -0.01872312 -0.05799608 0.08082969 0.018969342 0.02703605 0.29227605 -0.19298494 0.093478255 0.16663049 -0.2165682 -0.57485217 0.014979784 0.41007236 -0.15550932 -0.35630858 0.2340925 0.38467216 0.44292867 -0.33115035 -0.11217132 0.19208382 -0.4705107 -0.5332692 0.1062841 0.65658116 0.53078824 0.3595533 -0.022678878 0.12727973 -0.36216703 0.055228785 0.6161264 -0.36969426 -0.14443845 0.37922415; -0.021610014 0.016673876 -0.4225761 0.29580176 0.2141387 -0.27916068 0.3225807 -0.0011041302 0.32085586 0.15375914 -0.40543517 -0.21196538 0.21265619 -0.25444555 -0.14338411 0.33993432 0.11516647 -0.22921893 0.3579646 -0.25077787 -0.31999204 0.36661053 0.42542252 0.46506697 0.3358001 0.50267434 0.13019738 0.05310503 0.2954739 0.3233826 0.21083264 0.27304158 0.36270234 -0.32630906 0.31282765 0.3208243 0.4152339 0.3559298 0.25645557 -0.5608399 -0.31029013 -0.4911079 0.43613425 -0.34310374 0.22754456 -0.110150605 -0.1596977 0.011073472 0.41499177 0.045322273 0.10802693 0.16284797 -0.42913404 -0.33560073 0.275799 0.37432468 0.09318987 -0.036672063 -0.3456104 0.15899205 -0.34666735 -0.21663705 -0.18163009 0.3230165 -0.08160551 -0.1674909 0.25058332 0.4422996 0.044163976 -0.3243816 -0.25636303 -0.15314285 -0.094940916 0.16555524 0.4379705 -0.096102856 -0.5079026 0.15736978 0.3757731 -0.37732163; -0.34167174 -0.09905121 0.2850254 -0.14294039 -0.3655668 0.04450197 0.312301 -0.1348002 0.11477991 -0.4868398 -0.008055666 0.5697432 -0.62002397 -0.1326237 0.19194102 0.077921964 -0.4910236 0.17373501 -0.26236454 0.017479185 0.23662943 -0.54969233 0.22959226 -0.059478406 -0.5083525 -0.1744887 -0.30753362 0.040302187 -0.3645376 -0.26222134 0.34448794 0.085674465 -0.43931088 0.45414105 -0.33540115 0.26430216 -0.09125381 0.14369254 0.15637703 0.43082115 0.23188403 0.2615055 -0.4278636 0.32565033 -0.16036905 0.4477835 -0.09448788 0.18401127 0.28987342 0.18612663 0.33030462 -0.12720026 0.47379377 0.19795156 -0.47875658 -0.46117312 0.3570645 0.34453684 -0.02181423 -0.1133155 -0.109554 0.39463603 0.2048155 0.1604388 -0.05802778 -0.08043328 -0.10423713 0.111918725 0.24164072 0.37937447 0.47150722 0.41742155 -0.38979316 -0.02071532 0.15381339 0.11333236 0.121096864 -0.36593294 -0.08897123 0.27033854; -0.17114903 -0.45415393 0.28419656 -0.24160163 -0.10104561 -0.07312658 0.10831005 -0.3594395 0.24736877 -0.14192519 0.09181248 0.57266706 -0.69307476 0.0444471 0.046382084 0.08279767 -0.21637687 0.2017215 -0.30102178 0.06452613 0.059338752 -0.23574318 0.036623072 0.0014056817 -0.30195907 -0.24054402 -0.25856274 0.10683754 -0.14296937 -0.4359826 0.12371612 0.2501642 -0.5521749 0.1012117 -0.08204464 0.0038407713 0.14509308 0.26074374 -0.03755569 0.46349734 0.36697912 0.4309217 -0.50719494 0.45780495 0.03396893 0.37472156 -0.047224365 0.07604961 -0.032876343 0.17919475 0.18427144 -0.19254449 0.17288522 -0.030637374 -0.25867188 -0.2878607 0.3971481 0.2077246 -0.18702061 0.06764258 0.1283216 0.24298087 0.5519697 -0.07863741 0.025000574 -0.036112905 0.0033875385 -0.071274966 0.18783413 0.47543964 0.2390897 0.3650592 -0.29111335 0.16176899 -0.07812388 0.4219564 0.2600146 -0.3335603 0.16145617 0.27434817; 0.19939092 0.04663603 0.16753332 -0.397006 -0.19886608 0.3547967 -0.45555484 0.037276417 -0.20058614 -0.30632663 0.42307037 -0.15526277 0.20654675 0.32690623 0.22020665 -0.36415732 -0.03924912 0.10690412 0.14749381 0.33408245 0.20600612 0.4079488 -0.37611547 -0.27583867 -0.054388955 -0.35508686 0.27144122 -0.40974772 -0.15798299 0.18668129 -0.480106 -0.26159248 -0.30000877 0.33566156 -0.4522652 -0.33744 -0.24299164 -0.44582158 -0.3602049 -0.052322548 0.18831691 0.31829783 -0.2542513 0.28747502 -0.3204051 -0.10005292 0.20856236 -0.33351046 -0.46165276 -0.4810921 -0.34307513 -0.45205992 0.04620487 0.3652578 0.35255218 -0.10979099 -0.22319445 -0.5973685 0.34520954 -0.47464752 0.3812094 0.09776196 -0.25326556 -0.27799976 0.043079548 0.3944492 -0.5074887 -0.1936647 -0.24054925 0.22624838 0.35341218 0.13007006 0.42948127 -0.27083847 -0.22348715 -0.003413899 0.3383339 -0.19061688 -0.4095645 0.05670742; -0.3706594 -0.32051235 0.42921668 -0.42951664 -0.32455206 0.09839507 0.1403915 -0.34461102 -0.09956774 -0.18734363 0.15222119 0.32323202 -0.48294953 0.17149276 -0.11617784 0.054922707 -0.327029 0.33976883 -0.2295172 0.08402082 0.3759331 -0.16284314 0.13443027 0.12769863 -0.30644464 -0.3587649 -0.41195908 0.36463764 -0.5292288 -0.35000572 0.117765345 -0.015294771 -0.6331106 0.599465 -0.1450672 0.1470898 0.15136838 0.13507833 0.098553486 0.5486406 0.47382352 0.18022797 -0.4947659 0.0073614023 -0.19717643 0.462751 -0.021116585 0.23276043 0.052793317 0.106576376 0.03015852 0.13821639 0.2746949 0.0100743165 -0.1952527 -0.4117627 -0.0030653968 0.15785044 -0.24258006 -0.20459753 -0.10210266 0.24606685 0.3586948 0.16899158 -0.3327043 -0.26204067 -0.075868145 0.024570847 0.35009232 0.545365 0.41534874 0.65545434 -0.31957453 0.1815132 -0.107415095 0.15309419 0.5104588 -0.29826483 -0.07322285 0.44239828; 0.13145362 0.28092492 -0.1830806 0.23215392 0.21954069 0.12960435 -0.022252714 0.2548448 -0.01927445 0.24313895 0.2829596 -0.5328914 0.43989503 -0.037517298 -0.030338453 -0.1211696 0.24276437 -0.52546483 0.24247994 0.14118764 -0.24498555 0.44004062 -0.3164047 -0.0923632 0.45442 0.19546494 0.3022808 -0.35761616 0.24342357 0.48826036 -0.40154418 -0.029494738 0.5368837 -0.42411697 0.287171 -0.24297029 -0.15853146 -0.26164025 -0.25268328 -0.535987 -0.2691916 -0.07286596 0.40356857 -0.1186602 -0.094064616 -0.2785657 0.19643626 0.03118572 -0.14563844 -0.009356403 -0.15825671 -0.048962537 -0.46990874 -0.000780249 0.47343385 0.32274622 -0.33822465 -0.5206743 0.107945405 0.019125242 -0.020432489 -0.18905176 -0.4712686 -0.28108615 -0.020566696 0.04134625 0.13742287 0.04076574 -0.2467507 -0.50031495 -0.2897041 -0.5286717 0.49944982 -0.13774575 -0.34298557 -0.37897193 -0.120136335 0.16343908 -0.14288968 -0.24523522; 0.45961174 0.22755873 -0.1720512 -0.1466789 -0.18861468 0.38164225 -0.2863769 0.14470828 -0.42994353 -0.058505967 0.15442443 -0.21137357 0.3749356 0.34812668 0.1154817 -0.18731415 -0.012519191 -0.067816004 0.35748062 0.32962677 0.27287892 0.5855523 -0.46340016 -0.39383852 0.24695806 -0.10296629 0.014806361 -0.45957246 -0.14543895 0.07363939 -0.21894291 -0.43089488 0.37635204 0.1126895 -0.47579855 -0.31941086 -0.4552753 -0.4909982 -0.37809825 -0.071183816 0.197238 0.18003783 0.20516628 0.0037022242 -0.46351022 -0.19621955 0.17074797 -0.27383092 -0.21231467 -0.20720272 -0.32937488 -0.50511956 -0.12071678 0.47040638 0.3657822 -0.10814578 -0.0046533286 -0.37754276 0.5066575 -0.3422037 0.4670523 0.018987179 -0.33198404 -0.2837175 0.16921428 0.3740689 -0.074104294 -0.40487248 -0.21353541 -0.27787617 0.4791036 -0.09805639 0.18975918 -0.06923194 -0.439956 -0.14004095 0.2681103 -0.35783076 -0.22555749 -0.028553417; -0.079819284 -0.21599719 0.44498563 -0.29736105 -0.27052924 0.19041148 -0.077735856 0.099690594 -0.42833716 -0.17109773 0.2996602 0.41896632 -0.40950614 0.14889866 0.4452883 -0.26189044 -0.27393764 0.4170042 -0.1538269 0.11123364 0.39953285 -0.2643313 -0.2233403 -0.44236973 -0.44877478 -0.47021708 -0.4159402 -0.048008073 -0.49605215 -0.40890428 0.085447274 -0.3232782 -0.21180002 0.41345078 -0.14987738 -0.13066588 -0.42543605 -0.2872445 -0.14231494 0.532158 0.17366414 0.13875897 -0.49998662 0.18886843 -0.2968572 0.42401952 0.071592025 -0.027577247 -0.093791135 0.18305734 -0.14296415 -0.15017827 0.34662005 0.24348457 -0.26624072 -0.4301035 -0.073565334 0.22028713 0.24431469 -0.12972215 0.3142053 0.3186007 0.18272796 -0.4694951 0.061500806 0.29716522 -0.42887062 -0.1491387 -0.00012642285 0.5034299 0.46111116 0.5018349 -0.28700814 -0.25047383 -0.2924464 0.105634175 0.38584217 -0.38387775 -0.21105155 0.33542207; -0.31811017 -0.33332816 0.40325147 -0.1446708 -0.17464565 -0.043622177 0.2432009 -0.19710506 0.14665799 -0.12820648 -0.27091998 0.2898483 -0.59595037 -0.13207826 -0.03619279 0.34058598 -0.3939249 0.40216255 -0.28967533 -0.1503567 0.216329 -0.2876133 0.3162722 0.07175571 -0.33865228 -0.3205839 -0.39812785 0.32094878 -0.12197532 -0.24698626 0.23751763 0.030356193 -0.6166227 0.019305313 -0.18791845 0.30997851 0.27195832 0.17082146 -0.03450238 0.3568245 0.3799662 0.10457238 -0.31557196 0.3361435 0.23313954 0.24672548 -0.1711825 -0.05067303 0.37349284 0.14250544 0.35665157 0.0717386 0.38907406 -0.07727735 -0.419656 -0.26149306 0.19871226 0.27206534 -0.24549958 -0.04620878 -0.058938105 0.298022 0.3676251 0.3002954 -0.026808433 -0.21383496 -0.04923568 -0.052322816 0.12404023 0.26945317 0.40450165 0.4224365 -0.44803998 0.26618445 0.27066684 0.1620104 0.15462169 -0.45994514 0.1809883 0.2023659; 0.23106124 0.4776868 -0.23833707 0.15914464 0.42805398 -0.27188 0.16001089 -0.16405647 0.3366311 0.5126421 -0.4505323 -0.2646202 0.41142213 -0.472954 -0.15895383 0.46720442 0.2356089 -0.18839884 0.338714 -0.14078045 -0.4498894 0.19936979 0.19395882 0.11659191 0.1758328 0.43953916 0.54541516 0.16668242 0.35989475 0.20925963 -0.07268803 0.44609168 0.42232653 -0.15979627 0.17915833 0.15802288 0.36254156 0.09758422 0.28306088 -0.31012478 -0.18513222 -0.35965827 0.34437773 -0.38240543 0.34681582 -0.23215285 -0.14896509 0.12441489 0.119950645 0.1200451 0.33300936 0.13299222 -0.23598482 -0.135614 0.06403995 0.4062814 -0.2422571 -0.3520538 -0.15375897 0.13050932 -0.13402328 -0.14720145 -0.22751614 0.34121677 -0.243992 -0.29003254 0.12721452 0.23038203 -0.21969691 -0.5024712 -0.47579086 -0.4384474 0.20248668 0.25687283 0.19671237 -0.30915305 -0.4199958 0.24056724 0.44824705 -0.24418516; -0.21126688 -0.21715997 -0.13715841 0.2762579 0.43088552 -0.52623785 0.5298602 -0.2928426 0.2158133 0.53065753 -0.46832606 0.103283584 -0.33964562 -0.5306146 -0.3460164 0.39183918 -0.11648888 -0.2730599 -0.41763347 -0.5284039 -0.23042114 -0.21971509 0.40856758 0.5436126 -0.2216685 0.56529903 -0.03657496 0.15087263 0.19812442 -0.049076665 0.36893958 0.27189472 0.07122341 -0.23479474 0.27067587 0.36882728 0.4193566 0.3847501 0.37081826 0.005473249 -0.5077359 -0.47245625 0.25664067 -0.17456165 0.65143013 -0.09393761 -0.07973153 0.11352934 0.54276085 0.50575376 0.4093345 0.4095424 0.006843961 -0.3403713 -0.117888905 0.45507106 0.20802854 0.30271244 -0.49926075 0.4902345 -0.3732124 0.09592188 0.19267747 0.4921318 0.050673064 -0.44160903 0.17569074 0.13112423 0.02016846 0.08243354 -0.32830656 -0.29398867 -0.47591177 0.31009343 0.26534778 0.15775116 -0.16824219 0.46968332 0.26807216 -0.017514467; 0.25713897 0.10958351 -0.2769891 0.29396906 0.45946962 -0.0016967355 -0.15173994 0.3526857 -0.07240512 0.5376204 -0.2738579 -0.36830598 0.13237792 0.013005196 -0.226513 -0.13828668 0.25399682 -0.31758136 0.320593 0.07273527 -0.64782757 0.36269397 -0.2217713 -0.12657578 0.54144967 0.27278945 0.33856916 -0.46884903 0.53263825 0.24329013 0.0077567054 -0.2218383 0.6316866 -0.59659094 0.16768456 0.08863442 0.2642229 -0.03043156 -0.2898469 -0.6949162 -0.5975169 -0.4014791 0.48272094 0.13488623 0.07133522 -0.14971723 0.34449488 -0.33553433 -0.16164474 -0.2611768 -0.12966911 -0.2724275 -0.3389954 0.26608777 0.5244783 0.7614954 0.22269084 -0.54481876 -0.005336871 0.10783517 0.13529016 -0.43316963 -0.23663838 -0.117741 0.3616307 0.022884673 0.17528783 0.23779675 0.053022645 -0.7367376 -0.2771492 -0.5980244 0.16056369 -0.17230015 0.040283397 0.19189933 -0.70022136 0.5665397 -0.19375555 -0.72083795; 0.35349247 0.1499285 0.30890277 -0.49955162 -0.54634714 0.42797956 -0.26653785 0.40682527 -0.23585226 -0.37946647 0.36508873 -0.19653273 0.26825738 0.43930086 0.25941426 -0.44474408 -0.11353877 0.34197333 0.27805725 0.3173191 0.33433646 0.3616007 -0.3019547 -0.29643598 0.041884046 -0.3870428 0.1755827 -0.16970643 -0.3261024 0.107013896 -0.21677549 -0.19142592 0.02944793 0.25193784 -0.3034545 -0.26032838 -0.2736446 -0.48739222 -0.30557036 -0.19190644 0.047082584 0.38060594 0.18828782 0.020250292 -0.28301683 -0.03858607 0.39877284 -0.31766713 -0.41280273 -0.2542058 -0.22415471 -0.4062567 0.012747292 0.2427081 0.3365162 -0.15813656 0.054043975 -0.45798486 0.2561246 -0.24959071 0.23141168 -0.079061635 -0.27856547 -0.43019423 0.3520802 0.31357792 -0.25685388 -0.1641246 -0.12263835 -0.12566847 0.3165221 0.0349176 0.33788595 -0.22631952 -0.2818591 0.044729643 0.3439243 -0.14014293 -0.48203355 0.18562499; -0.071316786 -0.38392356 0.19554026 -0.28727272 -0.29036093 0.11795748 -0.4433892 0.40815714 -0.25663283 -0.3109275 0.21345755 -0.059877794 -0.037197016 0.27112922 0.42016718 -0.09705567 -0.46984893 0.3447395 -0.29462314 0.24567951 0.12047373 0.17406 -0.5359503 -0.269003 -0.25571954 -0.073571265 0.05720041 -0.3548869 -0.08278375 -0.049781624 -0.009661228 -0.3731403 0.025721975 0.2910273 -0.21302295 -0.30154172 -0.049179237 -0.18295391 -0.5095063 0.081115484 0.36190665 0.29732677 -0.15578489 0.58059305 -0.25433996 0.12731682 0.4912377 -0.34842637 -0.34785995 -0.19083065 -0.34342206 -0.387176 0.3943025 0.25790066 0.55276376 -0.19981141 0.32601917 -0.31194595 0.462185 -0.39786106 0.42670363 0.2915444 0.13207357 -0.47974795 0.5267524 0.17445493 -0.012397786 -0.34382275 -0.0761437 -0.07953071 0.23449957 0.19076465 0.17500037 -0.57201344 -0.1462106 0.38703063 0.27862138 -0.16755687 -0.53962135 -0.10761982; 0.08211252 0.1793404 -0.39208183 0.12767544 -0.020652344 -0.020771159 -0.27598542 0.41014016 -0.1807245 0.49124038 0.16558751 -0.5328223 0.20852646 0.15154214 0.26606992 -0.14161895 0.2698176 -0.4245322 0.43468323 0.2824286 -0.15474555 0.15413004 -0.0664029 -0.075394586 0.62146276 0.35472503 0.43729287 -0.07334109 0.54541636 0.27166066 -0.41573974 -0.20555556 0.4177924 -0.7155706 0.16119511 -0.025200818 -0.24736191 -0.34297165 0.01594411 -0.54919636 -0.31238574 0.042466916 0.31500646 0.2299429 -0.07486214 -0.14056015 0.34273982 -0.3726785 -0.25646904 -0.12090352 -0.08501057 -0.09668679 -0.31359997 0.22508892 0.24705137 0.47640374 -0.04352676 -0.29261562 0.13454969 -0.0796684 0.30936682 -0.29930264 -0.4453819 -0.23910075 0.2720054 0.07413961 -0.109588094 0.041289885 -0.14172997 -0.2740008 -0.53393024 -0.44143334 0.0857891 -0.28286183 -0.17133404 -0.116048545 -0.43483996 0.5434337 0.0031122356 -0.5257087; 0.12473203 0.097543955 -0.30796278 0.3184735 0.35669062 -0.026918387 0.028881859 0.37084538 -0.24947155 0.50800765 -0.10981003 -0.41341126 0.30937937 0.001555455 0.20540524 -0.15438479 0.20627703 -0.5502657 0.34996498 -0.04321823 -0.37990645 0.33290905 -0.1036774 -0.3505718 0.48202026 0.5686049 0.3992602 -0.13377075 0.46600187 0.19595693 -0.42639434 -0.022002103 0.57448834 -0.672765 0.25386304 -0.31842616 -0.10625203 -0.041032963 -0.074338004 -0.6106666 -0.5748907 -0.28089264 0.27046388 -0.03233442 -0.08354045 -0.14895639 0.3420862 -0.09547375 -0.07459844 -0.09586276 -0.1291597 0.0330517 -0.42099908 0.21395485 0.438372 0.45746306 0.024176033 -0.5443231 0.326586 0.20356499 0.06601763 -0.37427545 -0.27408165 -0.24913037 0.4145838 0.2956484 -0.011261912 -0.09226679 -0.15561728 -0.4349286 -0.45669335 -0.5739068 0.24758878 -0.23399995 -0.22737566 -0.025144551 -0.46147472 0.3701839 -0.23262176 -0.5308962; -0.13342138 -0.16894354 0.21991481 0.28449604 0.17434646 -0.34427288 0.39875638 -0.06969186 0.43499336 0.121309355 -0.19284904 0.42545995 -0.24465342 -0.36429483 -0.43170947 0.15932903 -0.13502048 0.07987033 -0.21173999 -0.36596698 0.009852517 -0.5212825 0.19372508 0.42336097 -0.31842673 0.13501142 -0.29961032 0.13654268 0.1674391 -0.33968124 0.22051536 0.4569647 -0.35787037 0.15774156 0.41982582 0.3704001 0.10206485 0.26691064 0.48494208 0.45680848 -0.12194838 -0.25273603 0.04244327 0.08255354 0.37922212 0.009890975 -0.06745223 0.27256733 0.41809326 0.29593778 0.43867263 0.2573122 0.11154641 -0.18049707 -0.48637095 -0.2122305 -0.033570573 0.23030409 -0.15241453 0.49736512 -0.4902977 0.26353052 0.30043146 0.15399365 -0.10234895 -0.27415112 0.119839475 0.088351965 0.2630977 0.20452355 0.08776162 0.39742485 -0.4786092 0.44041073 0.447736 0.15755554 -0.18675596 -0.032169476 0.1940109 -0.03417863; 0.30195248 0.07797991 -0.4549593 0.30505294 0.22883858 0.30392522 -0.13224551 0.20000972 -0.062472943 0.2600879 0.2567801 -0.27732572 0.5152713 0.19143413 0.25807253 -0.1495013 0.1813609 -0.44791165 0.5942886 -0.015483888 0.0031635077 0.39680952 -0.17427142 -0.31038308 0.32185474 0.12658826 0.3925371 -0.18220578 0.08154137 0.4990153 -0.3705299 -0.08413614 0.2918204 -0.0653694 -0.11424659 -0.28512514 -0.22199932 -0.2956424 -0.3440575 -0.5328471 -0.15416838 0.005463016 0.29880595 -0.18862163 -0.22080554 -0.24488378 0.21638207 -0.23371737 -0.37857205 -0.46070728 0.025228187 -0.30189013 -0.37716722 0.28471836 0.3524571 0.022170978 -0.2268253 -0.49337834 0.099221945 -0.14345005 0.10983858 -0.3208943 -0.45693618 -0.12899438 -0.08517978 0.3015581 -0.3431492 -0.15543021 -0.27713764 -0.24546413 -0.36324978 -0.4146263 0.13644305 -0.15326044 -0.10211973 -0.1295065 0.20914029 0.12311449 0.030345175 -0.2795394; 0.09655711 -0.06609458 0.22725265 -0.44665235 -0.30101135 0.5406219 -0.22444715 -0.16809472 -0.37724435 -0.43022797 0.4037974 0.28854707 -0.0066199615 0.34260714 0.38012514 -0.12258125 -0.34437248 0.46840203 -0.08147857 0.40302092 0.551585 -0.23116079 -0.3747405 -0.4531439 -0.09475398 -0.57601947 -0.04274049 0.06890124 -0.5538821 -0.11315028 -0.22290173 -0.37665 -0.54843813 0.30753767 -0.35110506 -0.16192238 -0.31138313 -0.32810727 -0.37681773 0.473804 0.2572416 0.26113942 -0.61100334 0.18171138 -0.18428779 0.2868858 0.11894248 0.09421864 -0.15052772 0.24325453 0.04359438 -0.15649359 0.10915269 0.12584111 -0.17411116 -0.36245146 -0.18304847 0.27941537 0.3804275 -0.19079906 0.29141018 0.3537784 0.14444125 -0.3841716 -0.1539103 0.19491999 -0.43188718 -0.41363564 -0.08514114 0.5200049 0.5099712 0.28179237 0.1007181 -0.15791936 -0.23963702 0.1346247 0.55351454 -0.53684676 -0.19147633 0.5723389; -0.119731195 -0.42567325 -0.10287119 0.08530253 0.31053618 -0.4727563 0.28836206 -0.38388774 0.17927478 0.3309836 -0.4957613 0.16182436 -0.52102244 -0.32621148 -0.33411014 0.36127228 0.0636237 -0.11560932 -0.66123265 -0.5572359 -0.35070813 -0.37730947 0.47734228 0.16623285 -0.50279444 0.35580236 -0.29391947 0.41519833 0.13667366 -0.32985866 0.49840832 0.5065337 -0.22684546 -0.3341076 0.44520882 0.24792077 0.40697423 0.28980723 0.34985092 0.07630626 -0.21824624 -0.22037143 -0.13591263 -0.060692023 0.6928006 0.047029 -0.326071 0.17797172 0.15676416 0.23122947 0.55523056 0.48635575 0.08580944 -0.4062435 -0.3776931 0.070056535 0.35500816 0.30345345 -0.17602704 0.36386472 -0.27629682 -0.046318583 0.46100223 0.17311192 -0.06254142 -0.20051327 0.15143049 0.4012725 0.27394968 0.2714655 -0.56460625 0.09840081 -0.30747706 0.11020205 0.25334528 0.0059559485 -0.32899612 0.3803901 0.29154387 -0.081163816; -0.17944379 -0.32579878 0.46107262 -0.3968029 -0.3404658 -0.29417443 0.06621424 -0.3444388 0.04132763 -0.20213962 -0.10757837 0.43960443 -0.6732549 0.07170924 -0.2552235 0.22896862 -0.16028544 0.44821283 -0.28191862 -0.19935395 -0.047357008 -0.43089947 0.06621305 0.367931 -0.4909234 -0.18297154 -0.41598555 0.3180368 -0.06421824 -0.26178354 0.36541736 0.12349575 -0.3349663 0.07227851 -0.07279765 0.06437324 0.0025313676 0.25847456 -0.0015559483 0.28852504 0.36541772 0.044071507 -0.29844293 0.19459648 -0.12765726 0.3738863 -0.1993738 -0.032323774 0.3707975 0.10179859 0.24597359 0.18584742 0.15154693 -0.28500387 -0.36671305 -0.35211465 0.2092034 0.3987643 -0.06416995 0.06171888 -0.22754245 0.24123184 0.22540784 0.31620282 0.08548733 -0.18462892 -0.019584252 0.11577229 0.3951483 0.22670633 0.33426917 0.4204653 -0.21383825 0.1826268 -0.017067263 0.2794051 0.22593737 -0.31764674 -0.018169295 0.28762946; -0.46190095 -0.2780568 0.16902687 -0.4593652 -0.35375983 -0.10994988 0.016375622 -0.008187147 0.051246334 -0.17449024 0.104444124 0.49451765 -0.34241384 -0.1370933 -0.081303574 0.28617144 -0.26725432 0.26547626 -0.42728838 -0.07768056 0.054654416 -0.36530071 -0.03638085 0.24059361 -0.54042125 -0.24782999 -0.25152674 0.059638485 -0.46879607 -0.4418545 0.043442503 0.20864418 -0.5619506 0.17880838 -0.18396768 0.2964617 0.05491509 0.27687475 -0.24803929 0.371873 0.16154996 0.11456868 -0.3361153 0.2525811 -0.10569821 0.5195514 0.030854901 0.014598152 0.08703956 0.06626375 0.08572766 -0.15169306 0.24243586 -0.051247634 -0.03899548 -0.21186385 0.12952237 0.44211066 -0.016295407 -0.04312713 -0.090857506 0.2993714 0.52905583 -0.08680891 0.2898224 -0.06347106 -0.2898359 -0.07695186 0.1749787 0.4010015 0.30078888 0.2924038 -0.10162762 0.23402707 0.16102757 0.26810172 0.0729072 -0.36008134 0.15083452 0.35794562; -0.21422264 -0.07852281 0.18124989 0.29354757 0.31145203 -0.3442433 0.17759216 -0.29844806 0.17546211 0.20613311 -0.31996328 0.17114094 -0.4399357 -0.53384644 -0.4550774 0.4281421 -0.1205217 -0.20977348 -0.30961397 -0.2540758 -0.32938102 -0.4533094 0.3307869 0.27492574 -0.23447399 0.1902079 -0.08448133 0.4486131 0.09519289 -0.37188485 0.16174316 0.20126048 -0.2563107 -0.29655263 0.17901136 0.34382078 0.36631075 0.55196434 0.60452354 0.11455868 -0.1841615 -0.34863812 -0.09940352 -0.09567393 0.33401328 0.22455806 -0.177265 0.11077112 0.5222894 0.3903565 0.28629068 0.26520205 -0.13563715 -0.5227387 -0.3629878 0.060468365 0.027293012 0.2583809 -0.2911782 0.47490314 -0.30825457 -0.07765022 0.06675012 0.20917289 -0.11937308 -0.38050607 0.19455719 0.31583092 0.4169841 0.01366134 -0.10386959 -0.104088455 -0.15640077 0.2625116 0.40198243 0.19597913 -0.3492548 0.09936151 0.18821917 0.1612557; 0.05273545 -0.13609998 0.5446253 -0.28052682 -0.45764667 0.533609 -0.06278255 -0.28080612 -0.31244442 -0.47829017 0.56649435 0.60268235 -0.20016178 0.33060333 0.49952596 -0.4274015 -0.24171363 0.5621677 -0.20535518 0.3405049 0.23573972 -0.2592685 -0.28668746 -0.17032714 -0.42080504 -0.49841958 -0.06651005 -0.05230656 -0.35344714 -0.12119436 -0.095586956 -0.07502706 -0.39759955 0.31795698 -0.21144219 -0.2469866 -0.48334298 -0.12635584 -0.43362093 0.3536872 0.2332274 0.46828425 -0.5343019 0.11480206 -0.3365728 0.14890495 -0.21382955 -0.17200293 -0.12471517 0.26526096 -0.07009418 -0.38638458 0.20256495 0.154436 -0.46139255 -0.60348207 -0.081513934 0.27841595 0.2903245 -0.25987136 0.36214787 0.4944997 0.28215092 -0.24652202 -0.05125959 0.27045122 -0.5937486 -0.44471937 0.17884174 0.38007563 0.261824 0.47791284 -0.22723004 -0.00876924 -0.33255997 0.12418591 0.53218293 -0.2939691 -0.09938317 0.385932; 0.1173731 -0.12570937 0.31649262 -0.22070734 -0.22535264 0.44045684 -0.29247043 0.24913435 -0.4647437 -0.1786373 0.27659166 -0.09810207 0.33176816 0.45618507 0.43180647 -0.43903142 -0.05927274 -0.037224155 0.19333224 0.54068923 0.42237133 0.368119 -0.19036824 -0.20165047 -0.1991204 -0.10641476 0.15309699 -0.40243524 -0.2742799 -0.025664033 -0.29744932 -0.24518798 0.17092006 0.16104679 -0.22101235 -0.44928104 -0.15789771 -0.22870648 -0.3829463 -0.08213528 0.19243257 0.30190095 0.29243255 0.286404 -0.31741685 -0.21143104 0.40885484 -0.55363226 -0.19916783 -0.15280274 -0.35779357 -0.5878401 0.08298161 0.28374723 0.4734888 -0.04899951 -0.021093104 -0.24869929 0.2011848 -0.5350242 0.49803308 -0.09718847 -0.14119335 -0.17492355 0.20290491 0.34089524 -0.3508471 -0.20184016 -0.22692104 -0.35801226 0.27927065 0.053321883 0.20317045 -0.26860985 -0.35346907 0.010751236 0.44360447 -0.31851944 -0.25030643 -0.060588617; 0.26626092 0.20372495 -0.41826344 0.24141356 0.2358029 -0.19465676 0.32565972 0.13377889 0.4027577 0.24039729 -0.37751138 -0.25205088 0.44611305 -0.3418773 -0.18617254 0.42847428 0.3713604 -0.18510148 0.50550437 -0.23382515 -0.4299788 0.30323023 0.11296898 0.4300684 0.3927289 0.31486177 0.33771655 0.079519585 0.3608242 0.17499857 0.04086419 0.35043648 0.49670202 -0.44826993 0.15742205 0.39572582 0.5223071 0.3819218 0.33804002 -0.49482998 -0.24092843 -0.15647148 0.5461505 -0.40227264 0.2370294 -0.27787548 -0.033531398 0.008510691 0.32592353 0.040210154 0.14957182 0.27446806 -0.109917365 -0.21056737 0.47034007 0.47268972 0.06963794 -0.47960213 -0.006976438 0.30976495 -0.3730358 -0.23047039 -0.34875995 0.30089697 -0.18857214 -0.24328485 0.52552986 0.26533747 0.011100284 -0.5352661 -0.38449284 -0.38377059 -0.06843596 0.037823107 0.3391847 -0.1406539 -0.22526932 0.27442873 0.39005002 -0.46595117; 0.34666443 0.5118423 -0.4654628 0.13951735 0.46074197 -0.15788776 -0.025184838 -0.05037477 0.14767848 0.3725323 -0.06156609 -0.40361422 0.38895974 -0.27291286 -0.263583 0.04402309 0.36226916 -0.25727162 0.22085372 -0.03272393 -0.4284838 0.4421763 0.058823094 -0.05187229 0.28552735 0.27618372 0.49070022 -0.1365935 0.51447827 0.20009293 -0.10998602 -0.18014763 0.31233656 -0.21348129 0.40126035 -0.26607394 0.118861854 -0.21083671 0.05013624 -0.50225663 -0.30087858 -0.38753462 0.4103344 -0.121680625 0.3240014 -0.2548273 0.21176146 0.09337855 -0.11287512 -0.094466284 -0.070972346 0.043249425 -0.3924227 -0.0008175019 0.085751615 0.4068052 -0.2401978 -0.3285687 -0.0049968627 0.2781082 -0.10572981 -0.18754183 -0.53608084 0.18334876 -0.043534193 -0.078654155 0.4103525 0.22971945 -0.2012163 -0.29900908 -0.23824616 -0.19248484 0.039576925 -0.13697775 -0.14046265 -0.17788146 -0.2332769 0.37736505 -0.033755988 -0.29604167; -0.14615747 -0.16568421 0.2506423 -0.15954171 -0.39128643 0.25834164 -0.11534236 -0.3261442 -0.30342734 -0.30712143 0.33070198 0.35513717 -0.5355369 0.37701046 0.09874405 0.10014295 -0.33196998 0.2774852 -0.33698165 0.32680443 0.45434663 -0.41360536 -0.15515137 -0.09522423 -0.242046 -0.29972082 -0.19150257 0.042090848 -0.46131647 -0.40485007 0.20911792 -0.15855989 -0.26243597 0.36132228 -0.16443466 -0.03145434 -0.14656416 0.18086922 -0.09918893 0.44508547 0.20394176 0.13172989 -0.606865 0.2728159 -0.2831293 0.1569584 -0.1353288 -0.029955927 0.07547782 0.24133657 -0.1750421 -0.11428047 0.48696524 0.33571368 -0.38249 -0.44916686 -0.06748635 0.25125754 -0.15846822 -0.105987325 0.10952585 0.33485126 0.31013307 -0.12853357 0.057374906 0.22924356 -0.23117685 -0.20845705 0.24669565 0.6164473 0.19919243 0.4985359 -0.21898235 -0.20971799 -0.12145729 0.075262666 0.36697796 -0.4533543 -0.19674248 0.56790245; -0.09484291 -0.17671233 0.15090396 -0.22558185 -0.38892117 0.32716265 -0.14395636 0.12108468 -0.09667118 -0.39340568 0.283675 0.21286094 -0.3648104 0.08259036 0.33694732 -0.06304712 -0.12034172 0.29986984 -0.51077014 0.13168086 0.36949217 -0.22472484 -0.25723648 -0.3619648 -0.4188696 -0.15804036 -0.19066614 -0.12474389 -0.4466437 -0.4701565 -0.016653936 -0.22715355 -0.4640349 0.5401137 -0.19830827 -0.3483497 -0.46026948 -0.33417812 -0.44129157 0.5586857 0.5270535 0.25494796 -0.5066487 0.37038937 -0.2948285 0.15355377 -0.01255173 -0.08694043 -0.20290999 0.0069889594 0.054004125 -0.28779265 0.33979622 0.12000524 -0.43093064 -0.34232 0.1599184 0.43621302 0.3047214 -0.2147461 0.13096404 0.22771713 0.3405484 -0.23576021 0.03299925 0.25572857 -0.50953406 -0.42031506 -0.05854498 0.55877984 0.17145798 0.25879276 -0.23524857 -0.20356637 -0.21739784 -0.03981758 0.17259315 -0.15707375 -0.092013665 0.4923066; 0.24617228 0.23905352 -0.50861675 0.38413635 0.16711836 -0.07180995 -0.28557557 0.11355562 0.008993736 0.28535223 -0.27587116 -0.31190944 0.16025971 0.21238586 -0.14347017 -0.089800656 0.38473627 -0.3565764 0.3862984 0.30875424 -0.5305626 0.50568295 0.07369602 0.035442166 0.639809 0.41117847 0.04727782 -0.45398724 0.8171654 0.28961003 -0.26657182 -0.25126573 0.46710256 -0.607648 0.11865964 -0.19656093 -0.03585303 0.06755371 0.03297568 -0.3609265 -0.24537337 -0.12952958 0.47992435 0.1770243 0.059261244 -0.3029671 0.094808094 -0.30190837 -0.13670637 -0.18469332 -0.41185856 -0.31893432 -0.09453287 0.103429906 0.39916533 0.82608885 -0.15307675 -0.5209342 0.12470383 -0.25556603 0.23552404 -0.47515494 -0.45567733 -0.055308 0.5440203 -0.08255487 0.09347643 0.050282825 -0.031946257 -0.46436134 -0.5149287 -0.65486217 0.32144323 -0.19046828 0.0018655047 -0.014927054 -0.5084893 0.3196412 -0.14556694 -0.6316979; -0.05594321 -0.061860375 -0.08922645 0.10431226 0.36437404 -0.45611173 0.4049961 -0.28849447 0.52878916 0.16216011 -0.51460975 0.19122256 -0.025291774 -0.4711022 -0.24795194 0.41558883 0.21695793 -0.20391788 -0.048669115 -0.39890283 -0.26568294 -0.45416042 0.5763345 0.26960936 0.029082809 0.21960904 -0.079572186 0.46485922 0.25601047 -0.08594806 0.25762394 0.30917403 0.020017717 -0.16783361 0.23389713 0.30253556 0.467779 0.5283158 0.5581478 0.0950424 -0.35354963 -0.33189574 0.16637002 -0.057658087 0.23614192 -0.08407038 -0.18401365 0.257773 0.47533306 0.38933903 0.34903696 0.278097 -0.21437739 -0.44540846 -0.4655173 0.26437503 0.15095013 0.37630826 -0.38136294 0.26549208 -0.49752092 -0.098547846 0.17667843 0.3355058 -0.014252479 -0.51862174 0.2597962 0.29924172 0.25088745 0.033610433 -0.21496186 -0.009412507 -0.2500431 0.23172884 0.38918307 0.008949802 -0.20035253 0.36559105 0.40577188 -0.29201934; 0.13573308 0.22914638 -0.32694885 0.25109655 0.23658198 0.11574237 -0.18740945 0.15705317 -0.22415812 0.33843353 0.09418782 -0.33266136 0.555813 0.05232854 0.18906073 -0.24083741 0.39951965 -0.38250488 0.4558734 -0.13336389 0.024384163 0.4727118 -0.09691977 -0.37464508 0.26309717 0.17751174 0.26718518 -0.15246744 0.08215297 0.32720265 -0.15473494 -0.061302107 0.42624104 -0.29774144 0.16961344 -0.36693326 -0.0076825274 -0.27883667 -0.027733266 -0.5231564 -0.34919608 -0.33711725 0.1041149 -0.10585965 -0.018958965 -0.43447697 -0.07244506 0.07050883 -0.106032826 -0.27021047 -0.027477097 0.23807003 -0.13266516 0.19471656 0.14300932 0.1400191 -0.1197355 -0.29939806 0.24427705 0.17348778 0.17888111 -0.36964065 -0.23073286 -0.3239726 -0.19200186 -0.014416025 0.11256008 -0.09996674 -0.34411103 -0.19232462 -0.25783467 -0.24996711 0.21863978 0.09225164 -0.12627481 -0.35416812 -0.24045953 0.18380442 -0.29976693 -0.27376327; 0.09481758 0.20987754 -0.44315442 0.3019675 0.42210656 -0.37834287 0.07275683 -0.069451645 0.32316178 0.49500608 -0.23989245 -0.4798493 0.039390754 -0.10979956 -0.4391067 0.3267318 0.1267744 -0.5043566 0.42734015 -0.14043951 -0.26493412 0.22066376 0.3742726 0.14316747 0.19725531 0.35441417 0.06986709 -0.2705925 0.32014042 0.009535058 0.18338437 0.21530904 0.50353456 -0.43337765 0.338734 0.12893581 0.41337284 0.22875473 0.46045646 -0.5727156 -0.33030638 -0.2852127 0.47444153 -0.16506717 0.28858492 -0.2010029 0.22698906 0.03566994 0.066789985 -0.13611397 -0.08112206 0.31163144 -0.26174867 -0.16711836 0.2889659 0.38325548 -0.10173115 -0.16431981 -0.29235435 0.4164182 -0.48135948 -0.21289578 -0.33711806 0.41513297 -0.06613027 -0.42189837 0.5940276 0.5228906 -0.04317006 -0.5405084 -0.47934073 -0.5816676 0.11651067 0.1505771 0.5078684 0.04610332 -0.58338726 0.43038556 0.24594228 -0.58003503; -0.22399223 -0.14800362 0.36312047 0.11251671 -0.0034746372 -0.35164472 0.44770384 -0.10869298 0.1939943 0.21442051 -0.27628312 0.31489983 -0.41662717 -0.2232468 -0.042460095 0.31694323 -0.24825199 0.2786592 -0.34216252 -0.3370704 -0.33091745 -0.48409557 0.031049235 0.20341277 -0.3840523 0.23402515 -0.49108613 0.1489486 0.19180183 -0.15220541 0.22346628 0.2451873 -0.2508008 -0.2657118 0.052132357 0.31383604 0.3880803 0.24928473 0.3720771 0.36206025 0.08143054 0.03458536 -0.3233425 0.2791852 0.17398092 0.076303184 -0.35220736 0.06337486 0.30687436 0.11532157 0.123662144 0.11871834 0.020407973 -0.17959556 -0.2904291 -0.13513216 0.13151857 0.32295123 -0.4929386 -0.05570286 -0.07444101 0.049955316 0.36152625 0.22366446 0.2433665 -0.46227077 0.0033494579 0.35434735 0.10487005 0.0468008 -0.1305312 0.10576078 -0.30368486 0.30489957 0.52081805 0.017547943 -0.1573626 0.019632867 0.13761656 0.20773491; -0.21294552 -0.12772262 0.24108398 -0.4100015 -0.4526354 0.2741961 0.1783916 -0.32862037 0.03393182 -0.44533053 0.3289828 0.30100608 -0.26910567 0.20543687 0.08476707 0.098067656 -0.23092934 0.49614272 -0.23102029 0.11715186 0.27447912 -0.27897793 0.12138676 0.10323592 -0.48223644 -0.5867307 -0.03924468 0.30367067 -0.43401337 -0.43082616 0.12788501 -0.049592216 -0.5650925 0.3212011 -0.42797717 -0.1081602 -0.33686975 -0.090208754 -0.039250664 0.5174508 0.4368862 0.13287924 -0.5180661 0.20818074 -0.13065057 0.14883587 -0.2935812 -0.027933711 -0.070069596 0.24631692 0.18945174 -0.032464262 0.4306545 0.17951748 -0.5767069 -0.5418705 -0.16972691 0.13268587 -0.250314 0.030721378 -0.09808246 0.31989902 0.34445983 -0.21563295 -0.1383487 -0.0035921852 -0.19806834 -0.24349985 0.06783549 0.71114624 0.40762302 0.4629744 -0.01329189 0.040337212 0.019359846 -0.12815292 0.6251791 -0.4244922 0.052565854 0.7445441; 0.055448886 -0.15363991 0.46382546 -0.18808122 -0.30962467 0.18098426 -0.21507905 -0.056514677 -0.38543802 -0.38492805 0.48713434 0.45535582 -0.36854863 0.24793734 0.4804948 -0.28147742 -0.31651908 0.32421926 -0.34960213 0.3470477 0.44998452 -0.40659037 -0.2021198 -0.14471644 -0.3353779 -0.28920427 -0.06089669 0.18795966 -0.29918063 -0.1733589 -0.04216898 -0.11689658 -0.55610657 0.5005969 -0.4663674 -0.30022797 -0.52527964 -0.07487473 -0.26400894 0.25592577 0.24995363 0.21492924 -0.56472075 -0.02701583 -0.5070245 0.066649966 -0.13965468 -0.27605444 -0.32239598 0.088156745 -0.008495285 -0.39674878 0.33753267 0.18770081 -0.1974986 -0.45146245 -0.04935125 0.2900033 0.008365702 -0.3503097 0.2064031 0.4360765 0.4239409 -0.40861985 0.050349005 0.17705911 -0.39273277 -0.34417787 0.1248773 0.5398656 0.29540884 0.34993008 0.05491037 -0.12357083 -0.11849853 0.24083063 0.26209658 -0.36581331 -0.36983 0.52870864; 0.23261501 0.016337268 -0.24221866 0.14876579 0.3031736 -0.38080087 -0.020173352 0.34440485 0.08912238 0.5259755 -0.50446415 -0.5999781 0.15166129 -0.24797074 -0.45961216 0.19448642 0.44100425 -0.31499714 0.23815548 -0.28517652 -0.42790234 0.2983479 0.35118046 0.30344415 0.36895934 0.4662537 0.31176284 0.018506624 0.61101663 0.17627364 0.17758128 0.22801341 0.59115505 -0.4164563 0.28957286 0.21206336 0.28420988 0.25878266 0.3115303 -0.3835741 -0.356318 -0.17922635 0.34073803 -0.055831693 0.2776095 -0.05967756 -0.0028379182 0.23526552 0.2869993 -0.19679874 0.08940841 0.3413947 -0.1752103 -0.38765812 0.27386072 0.4254229 0.06763112 -0.20687082 0.011146778 0.17992148 -0.21883395 -0.5100632 -0.4150598 0.13034523 0.025256738 -0.28935003 0.5951191 0.22608109 0.0650416 -0.68005574 -0.22147143 -0.43641427 0.012055565 0.1906026 0.37019217 -0.23481429 -0.35579073 0.34945467 0.37295696 -0.65104884; -0.2584788 0.05058718 -0.32814378 0.22378023 0.29203162 -0.28927466 0.44757873 -0.08276071 0.4904633 0.2353489 -0.48921758 0.07669787 -0.53887105 -0.5548447 -0.35481226 0.25607184 0.20711088 -0.21844317 -0.4238789 -0.30530038 -0.20107412 -0.20723774 0.4906613 0.46048382 0.057787646 0.47326607 -0.05541563 0.08756041 0.37732238 -0.3099191 0.29312786 0.24944222 0.17107375 -0.32847086 0.38937908 0.46583194 0.17013527 0.4410191 0.4111509 0.058015145 -0.29443896 -0.38696828 0.2172859 0.048420105 0.6454928 -0.122153506 -0.05275397 0.2535657 0.4779599 0.32441407 0.30885988 0.41828203 -0.17303424 -0.27397206 -0.1819986 0.170835 0.11875551 0.23327781 -0.5119826 0.560129 -0.2950257 -0.093956135 0.3059437 0.4143765 0.013710587 -0.49548998 0.29857543 0.45976058 0.17698532 -0.13265721 -0.4923583 -0.34444422 -0.291853 0.1352338 0.4879861 -0.004882673 -0.39975345 0.41486445 0.30269596 -0.14493047; 0.17707717 0.36612886 -0.19911595 0.33297807 0.25272906 -0.23952559 0.13671641 0.008333073 0.054227646 0.2911127 -0.31594038 -0.26794037 0.31422022 -0.38890272 -0.26189584 -0.019285494 0.15756598 -0.27277863 0.51770794 -0.16724522 -0.4927858 0.5004458 0.15120457 0.046556104 0.308569 0.49871606 0.39485252 0.007910139 0.2919434 0.17537037 -0.26169926 -0.03555862 0.24272892 -0.20612887 0.43947542 0.17904927 0.35314336 -0.026704665 0.13271278 -0.21423395 -0.27423453 -0.35879332 0.3777411 -0.16682255 0.436818 -0.122624226 0.11145085 -0.022123812 0.20319031 -0.25012675 0.02213609 0.26480427 -0.3368956 -0.04919909 0.34188014 0.35346502 -0.053199135 -0.26358646 0.10752556 0.121609256 -0.30245304 -0.1728054 -0.36852372 0.0011251005 -0.05753908 -0.35328776 0.38275367 0.13453327 -0.3022103 -0.6055442 -0.21956834 -0.20330532 0.23285289 -0.06657063 0.3929999 -0.07934301 -0.43420428 0.41283053 0.15326244 -0.28187552; 0.15534954 0.008982352 -0.35265845 0.3484616 0.38328904 -0.4600051 0.0054513975 -0.038345672 0.38759923 0.48559326 -0.21699592 -0.38754755 0.2413633 -0.38349822 -0.18910627 0.40895984 0.30704105 -0.5426572 0.42372242 -0.31507468 -0.50776184 0.29888996 0.4341883 0.09182742 0.24683706 0.42678055 0.1805113 -0.072042584 0.46876192 0.17066395 0.09722805 0.10577982 0.29956067 -0.5765573 0.33788174 0.073284365 0.3260729 0.3442128 0.20813607 -0.27724588 -0.28558093 -0.1568984 0.61826384 -0.16991997 0.3104139 -0.17061411 0.16723402 0.031066766 0.44866103 -0.24076961 0.27442297 0.29153755 -0.17400602 -0.39916486 0.3596541 0.5518344 -0.10195607 -0.2392153 -0.042207733 0.23556095 -0.50777066 -0.29107156 -0.33319885 0.3078093 0.18734504 -0.5093308 0.3072735 0.24815744 -0.0038735026 -0.6320094 -0.235273 -0.47871503 -0.12969647 0.226209 0.43668664 -0.03916764 -0.37104815 0.49187627 0.2358637 -0.624331; -0.36299607 -0.33714122 0.47228563 -0.17419037 -0.30766827 -0.1751959 0.061004877 0.044345886 0.021323605 -0.26001567 0.095518306 0.3807977 -0.5434639 0.0004522722 -0.07822718 -0.0077800374 -0.35073307 0.18336202 -0.3174852 0.2001799 0.21881656 -0.38764268 -0.0732482 0.05486391 -0.58485156 -0.48452157 -0.32728645 0.10445324 -0.2131165 -0.52288586 0.35034433 0.18722023 -0.25451687 0.43691358 -0.19355321 0.274867 -0.12967324 0.30426717 -0.013153896 0.28714582 0.15511775 0.1715171 -0.40410966 0.34820962 0.026956735 0.48014572 -0.00044254074 0.107115366 0.21907826 0.31617892 0.016345188 0.058780745 0.1498915 -0.006321171 -0.36856034 -0.30344573 0.15610908 0.26418915 -0.15026887 -0.040999923 -0.12850143 0.22304124 0.26220587 0.060166508 0.081311874 -0.13905802 -0.02078254 0.16087689 0.15606001 0.41805273 0.24894653 0.16686548 -0.22306015 0.03604279 0.21981181 0.2379075 0.2923314 -0.41890725 -0.07812724 0.46360365; 0.11216877 0.18576595 -0.1535126 0.41359684 0.20960504 -0.40519977 0.3630913 -0.09212653 0.4090296 0.16966148 -0.47840837 -0.5153188 0.21480411 -0.21837518 -0.35452846 0.12192341 0.46157137 -0.30274868 0.43621248 -0.12545927 -0.24727826 0.4167257 0.26757142 0.01442815 0.15975645 0.3119134 0.4739544 -0.2138214 0.39656773 0.37946603 0.056293536 0.11476387 0.21455628 -0.25811207 0.43040806 -0.00056015985 0.48151135 0.19974238 0.08399789 -0.2727304 -0.5298627 -0.26878512 0.298475 -0.11674982 0.11815619 -0.15852638 0.028577682 0.038934812 0.22703794 0.00757186 -4.6466776f-5 0.15401311 -0.11823727 -0.110369086 0.11749748 0.3192698 -0.14511056 -0.5058381 -0.2444361 0.13667685 -0.1456123 -0.50167596 -0.45141515 0.11675365 -0.02784701 -0.39798248 0.3967324 0.28930658 -0.03645414 -0.6315334 -0.14928922 -0.3994362 0.16571459 0.27932882 0.38445568 0.008055813 -0.5547491 0.166068 0.23862785 -0.595044; -0.034389272 -0.19112813 0.58013743 -0.4005994 -0.41518903 0.20080386 0.14757341 -0.26257044 -0.12073168 -0.41953292 0.1896359 0.40435603 -0.1453907 0.10623356 0.16767442 -0.08216252 -0.3652437 0.3631388 -0.17050946 -0.047291394 0.51622146 -0.4901719 0.005637471 -0.08090224 -0.18878093 -0.3225148 -0.1189857 0.15847479 -0.29048404 -0.38073173 0.26591522 -0.0846346 -0.3234921 0.3390945 -0.21415077 0.13090882 -0.350909 -0.019297535 -0.09559047 0.4467581 0.39783812 0.36041626 -0.6514051 0.0306995 -0.4195488 0.39941987 -0.25299683 0.14060098 0.063860334 0.21455035 0.28800645 0.011383742 0.22802967 0.02700005 -0.30641234 -0.65279436 0.09079702 0.38048327 -0.08863955 -0.35245794 0.124299675 0.3563998 0.4129897 -0.225315 -0.2815248 0.15537685 -0.19130659 -0.1307191 0.06566471 0.39872205 0.29002175 0.5254033 -0.28197247 0.034727953 -0.0061792824 0.08256086 0.3263495 -0.33793035 -0.20710559 0.44627178; 0.4772536 0.25380182 -0.21059683 0.27225083 0.4501371 -0.09637602 -0.16360722 0.1378211 0.1340192 0.29673946 -0.3029621 -0.25107318 0.43472484 -0.07023463 -0.1272649 -0.17310295 0.16451631 -0.24266222 0.48812303 -0.188205 -0.34760368 0.28509894 -0.0007427973 0.0643427 0.3020035 0.15173526 0.27077693 -0.0011315627 0.37649828 0.33151636 -0.15309921 -0.32160118 0.3396141 -0.3541336 0.39301583 -0.008696469 0.13156307 -0.12254986 0.06687469 -0.46157214 -0.30481014 -0.20838347 0.51268435 -0.17386758 0.0825052 -0.21993728 0.13843203 -0.15457839 -0.0820728 -0.028688094 -0.05028254 0.10585276 -0.3546438 -0.039377343 0.49087822 0.30151305 -0.2095221 -0.25021023 0.2403216 0.26477444 -0.06426929 -0.30823314 -0.54396856 0.17676634 -0.042603534 0.095236704 0.17900608 0.18803251 -0.22771682 -0.59720576 -0.1769427 -0.5683308 0.32553214 0.15454459 -0.22540618 -0.120549746 -0.359979 0.24856822 -0.03408325 -0.3322411; 0.08900275 0.2167277 0.29252666 -0.27463916 -0.4250565 0.32649687 -0.2613718 0.16827534 -0.542423 -0.23458572 0.25564092 -0.09019995 0.24534842 0.45358115 0.49572662 -0.42670748 -0.023410305 0.44017348 0.29466286 0.42961356 0.4138948 0.34004515 -0.34486595 -0.540313 -0.078411974 -0.15396379 0.092893705 -0.2816709 -0.15628901 0.10702709 -0.3971327 -0.29634032 0.12620312 0.3846139 -0.20707336 -0.29134956 -0.483924 -0.34413883 -0.31522712 0.04575098 0.17954436 0.17183784 -0.017195892 0.27070394 -0.42842704 -0.07757801 0.37858373 -0.37212265 -0.48173496 -0.30231607 -0.46155202 -0.58283603 0.16628708 0.26658 0.44743732 -0.28780824 -0.007543292 -0.558544 0.34452912 -0.21943095 0.2859839 0.24733683 -0.22214377 -0.44518805 0.12274669 0.54320025 -0.39400843 -0.2982499 -0.024063252 -0.028318789 0.16160241 0.35411674 0.416689 -0.33743382 -0.5489521 0.2610463 0.36660504 -0.2208187 -0.17924027 0.063615695; 0.052154955 -0.20019108 0.36115816 -0.3485694 -0.27639207 0.45224822 -0.039246224 -0.29241836 -0.20411424 -0.3196843 0.20877902 0.55748886 -0.13631442 0.08185759 0.41185364 -0.38516772 -0.475675 0.17993808 -0.5298394 0.21200004 0.29470623 -0.36365542 -0.19151105 -0.07484441 -0.4217314 -0.21947488 -0.21093257 -0.029615805 -0.49555182 -0.22365864 -0.10816386 0.025395297 -0.48675168 0.47099954 -0.21393207 -0.3011049 -0.47500148 -0.017493967 -0.33299935 0.26853415 0.53677946 0.4660456 -0.58741474 0.30871677 -0.17084417 0.33756578 0.0020179222 0.012421134 -0.3708109 0.29158914 -0.056299236 -0.1103478 0.45583987 0.16483778 -0.4066452 -0.25065336 0.13928801 0.2095679 0.07979237 -0.32637838 0.44083825 0.30681252 0.49407122 -0.062694766 0.22048168 0.20804414 -0.51999176 -0.47858474 -0.016788358 0.4426708 0.28348285 0.3393734 -0.102134585 -0.06797644 -0.23206393 0.1913656 0.37088245 -0.2739182 -0.21740513 0.32686642; 0.18734455 0.26621377 0.21184024 -0.4557708 -0.49176624 0.59425807 -0.27132156 0.24337353 -0.36568338 -0.5225301 0.42701948 0.094547644 0.14256108 0.40813807 0.52981406 -0.35200822 -0.077967584 0.45135015 0.0977545 0.5362584 0.27778685 0.2767971 -0.62135375 -0.5745296 -0.054544713 -0.24506648 0.14019804 -0.33758467 -0.16209736 0.29902127 -0.21809836 -0.56517285 -0.15034334 0.27447623 -0.35625076 -0.47150004 -0.40183088 -0.3797688 -0.4511729 0.1119935 0.5249547 0.3508695 -0.22586083 0.11490321 -0.65118426 0.16975893 0.32537478 -0.25095665 -0.47825882 -0.5482637 -0.28488034 -0.44473305 0.07356877 0.25752082 0.17885372 -0.38499844 0.030921066 -0.1664089 0.33936518 -0.27548224 0.43690413 0.17531255 -0.12708783 -0.47420272 0.13228558 0.47077936 -0.29243332 -0.28003755 0.035615202 0.22727785 0.5890442 0.4963861 0.07662266 -0.26076102 -0.5424385 -0.11793893 0.5189167 -0.28600228 -0.28044325 0.39799684; -0.24875444 -0.4140525 0.24076909 -0.15263109 -0.10771722 -0.120831124 0.1481797 -0.10622769 -0.08538569 -0.46661165 0.27105185 0.563645 -0.41540942 0.25672275 0.14487258 0.17410429 -0.33504954 0.4246489 -0.5040535 -0.017590765 0.44734764 -0.24826278 -0.09447348 -0.007008599 -0.44560727 -0.31871057 -0.203698 0.05867407 -0.41144156 -0.33581066 0.27212992 0.14211228 -0.3320103 0.2742731 -0.3429919 0.04585765 0.08186886 0.23736835 0.05017972 0.4544601 0.2431827 0.42120963 -0.42330098 0.20090534 -0.33569512 0.39874566 -0.32705453 0.10840394 0.2844771 0.31764075 0.032169133 -0.1960673 0.18607794 -0.04215967 -0.46311632 -0.51766443 0.03908 0.3031552 -0.012921133 0.03707622 -0.13621555 0.31892648 0.21686742 -0.14551714 0.038158208 -0.12931357 -0.11797658 0.0679021 0.30082732 0.42981935 0.2598168 0.42948216 -0.19261557 -0.052437294 0.10341921 0.137611 0.43502453 -0.23133588 0.19532944 0.5676589; 0.45757538 0.07048048 0.013800093 -0.24620265 -0.4025129 0.4240675 -0.4536435 0.088256516 -0.32457483 -0.24831353 0.35366225 -0.054431282 0.49608102 0.41583768 0.28258613 -0.31723267 -0.10203844 0.060819972 0.36653626 0.46862966 0.43657604 0.6131401 -0.48689914 -0.24696933 0.18212113 -0.45244688 0.12174373 -0.11141199 -0.35278815 0.17594725 -0.20894507 -0.5501988 -0.09368485 0.20911798 -0.34289867 -0.48810574 -0.13974324 -0.3389566 -0.3883957 0.0859488 0.11019868 0.35063672 0.16738977 0.11084099 -0.66046256 -0.26972696 0.10050633 -0.1844924 -0.38409048 -0.27899247 -0.51354975 -0.49328604 -0.10213919 0.30946392 0.11882511 -0.08101968 -0.23335804 -0.3218301 0.20262675 -0.39644203 0.28775072 -0.029513905 -0.07922301 -0.36308083 0.15352707 0.29253033 -0.47515702 -0.26275453 -0.12361684 -0.16657807 0.5697157 0.064378984 0.23635481 -0.24826895 -0.27999318 0.062262207 0.4343391 -0.3425996 -0.21836041 -0.0049537667; 0.1795288 0.2130759 -0.25621098 0.15440457 0.17334345 0.15985893 -0.071653455 0.048971605 -0.29860586 0.11654428 0.056239754 -0.47940686 0.63214827 0.10545208 -0.09028195 -0.061429564 0.45675054 -0.47810432 0.30889696 0.18958527 -0.1065052 0.36022943 -0.32378978 -0.36965844 0.5427094 0.29350236 0.25616163 -0.28179514 0.104172744 0.29582378 -0.4538114 -0.16931535 0.5082757 -0.19324222 0.22391628 -0.21022584 -0.020826392 -0.07895073 0.03741051 -0.49309894 -0.35511988 -0.033871356 0.18511766 -0.18634328 -0.13476384 -0.22958499 0.24445894 -0.08529059 -0.15620156 -0.03316832 -0.23180649 -0.031385396 -0.5083695 0.25642687 0.10082762 0.35473365 -0.45369264 -0.43011892 0.22973134 -0.017278925 -0.034312297 -0.41290253 -0.42723423 -0.07988781 -0.36098576 0.31559476 -0.07040315 -0.12580876 -0.36188158 -0.16697519 -0.38119468 -0.43186346 0.30556196 -0.013460484 -0.08732503 -0.40099868 0.027044898 0.20700222 -0.25476676 -0.15801683; -0.11176339 -0.16334179 0.2721776 -0.48357722 -0.5738147 0.25220302 -0.065097384 -0.18372516 -0.25050822 -0.56939465 0.25369754 0.40861872 -0.23674563 0.34580317 0.30595627 -0.07457093 -0.25440177 0.25641614 -0.19394615 0.34312835 0.48434693 -0.18031286 -0.37676954 -0.38190642 -0.31900087 -0.34458575 -0.06298216 -0.0868533 -0.56171244 -0.061032124 0.15193681 -0.10222223 -0.21339571 0.26886284 -0.22330551 -0.34064692 -0.07499867 -0.23451589 -0.4169375 0.5107764 0.5016698 0.20953932 -0.38677797 -0.011787137 -0.32031178 0.17012666 -0.042412315 -0.05312107 0.010244179 0.21228057 -0.246729 -0.39530692 0.09958253 0.24353412 -0.25849766 -0.52851295 -0.16637737 0.33157697 0.16086699 -0.35261756 0.23251016 0.30823502 0.2646406 -0.28711602 0.05232342 0.3371258 -0.41223508 -0.18675505 0.18727313 0.22290716 0.33290178 0.39137924 0.028648304 -0.12858565 -0.1018694 0.20387468 0.41160658 -0.30651176 -0.3654613 0.31241068; -0.37094998 -0.27077365 0.17496714 -0.37153304 -0.30502334 0.2673479 -0.0007606139 -0.05799696 -0.3008676 -0.20681928 0.1394166 0.248705 -0.40172848 0.2569045 0.14001326 -0.15673108 -0.41553813 0.3732414 -0.39089662 0.18407418 0.4357878 -0.16649435 -0.036088526 -0.2784587 -0.4020158 -0.29455176 -0.4896239 0.06210566 -0.3378404 -0.50348026 0.04232226 -0.0076947236 -0.5299303 0.24386121 -0.17859967 -0.30970433 -0.13762125 -0.27283913 -0.23454255 0.31156573 0.22057226 0.43885344 -0.5054002 0.20518315 -0.09392684 0.15995564 -0.023259059 -0.051889654 -0.08894572 0.082043454 -0.00012680303 -0.26844138 0.27663663 0.37028024 -0.21967168 -0.55248255 -0.12015968 0.37952134 0.13122614 -0.19387385 0.22587316 0.46749884 0.41871274 -0.34323108 0.16064304 0.12380334 -0.48026925 -0.34727803 0.27432162 0.49332005 0.13079405 0.47612906 -0.06796244 -0.35549414 -0.41423896 -0.01041706 0.3736023 -0.1322944 -0.12913942 0.4253627; 0.20998582 0.1217163 -0.5016537 0.29065442 0.3737979 -0.27024144 -0.00991644 0.07430251 0.038222704 0.4736789 -0.1647442 -0.61011004 0.50213283 -0.021762272 -0.23239596 -0.06449974 0.16764069 -0.5782672 0.46973118 -0.17799658 -0.46735415 0.1837395 -0.16147727 -0.046119038 0.36018094 0.24472861 0.24300466 -0.082893334 0.37171724 0.26145205 0.028667253 -0.026184255 0.40659598 -0.506237 0.2300185 -0.23999463 0.27743334 0.004626414 0.26161546 -0.3177885 -0.54575926 -0.48469114 0.35026708 -0.001625607 0.26681343 -0.20357892 0.4127005 0.059459243 -0.029114975 -0.1274577 -0.29911092 -0.11044844 -0.15598325 -0.20568714 0.550488 0.63939935 0.1513893 -0.379077 0.033972625 0.27043426 0.059937518 -0.5577888 -0.36654672 -0.16658925 0.37565932 0.16759056 0.5026481 0.05750083 0.022916911 -0.42001864 -0.33310896 -0.33655298 0.20714806 -0.075808294 -0.052849516 0.048542887 -0.6130581 0.46364614 -0.120102316 -0.50551903; 0.19623384 -0.06816692 0.16492695 -0.3796024 -0.30659375 0.25900066 -0.43087432 0.35503608 -0.39991352 -0.19755328 0.46504644 -0.058254585 -0.03297105 0.22779806 0.51536024 -0.5777517 -0.20852031 0.22384 0.17477588 0.22862686 0.38743278 0.29104534 -0.27763575 -0.29231954 -0.097954705 -0.23007478 0.10407544 -0.12181514 -0.39200303 0.0570592 -0.23012869 -0.29974857 -0.28896257 0.14855243 -0.32916546 -0.47011194 -0.22333787 -0.47588313 -0.46517822 -0.013555541 0.48214406 0.18008587 -0.08621291 -0.018331952 -0.49499485 -0.09106597 0.28192043 -0.3869182 -0.38503405 -0.46962044 -0.27181214 -0.4591382 0.01201108 0.32461753 0.39354175 -0.2830622 0.045840874 -0.24048199 0.42813507 -0.2629005 0.38702494 0.2624502 -0.03284551 -0.28958535 0.19087045 0.5126991 -0.48862794 -0.5098802 -0.021496143 0.054153245 0.43141595 0.43616313 0.3608174 -0.27223915 -0.55264443 0.033781998 0.40219182 -0.23821835 -0.23314032 0.29974943; -0.1174879 0.19386746 -0.22789828 0.34183583 0.22601233 -0.17197055 0.55926883 -0.32687655 0.18903142 0.31916112 -0.035253763 0.16525155 0.22072037 -0.49170896 -0.32879192 0.35958037 0.39177847 -0.45057142 -0.2060599 -0.46491048 -0.03523321 -0.07771489 0.5280839 0.0848661 0.1342283 0.28044152 0.04330391 0.32748076 0.13989697 0.0896315 -0.06967295 0.2985188 0.29631054 -0.08899678 0.4080151 0.44000927 0.2643482 0.30555472 0.60107 0.13692373 -0.2370196 -0.34800464 0.17214958 -0.23924735 0.45443928 -0.440202 -0.5297136 0.49159497 0.35468176 0.48603094 0.6103924 0.3815631 -0.1852807 -0.41765022 -0.54193634 -0.08360332 -0.34489244 0.22685395 -0.29719356 0.56118363 -0.39387706 -0.326834 -0.09647907 0.13606665 -0.35257462 -0.18322693 -0.010477124 0.2883854 -0.001942246 0.13134812 -0.18445559 -0.15445517 -0.24056207 0.6221686 0.19358343 -0.20889263 -0.042096615 0.25445923 0.23144594 0.15586148; 0.25796565 0.004501252 -0.5419639 0.32212687 0.49667588 -0.14777382 0.09731358 0.26603526 0.22300182 0.5268033 -0.2666566 -0.4180058 0.38156137 -0.08677756 -0.33844292 0.038591377 0.23343347 -0.30045938 0.43509054 -0.18516713 -0.5051792 0.3168224 0.1431061 0.30294698 0.46324238 0.565177 -0.005133335 -0.033665366 0.50078654 0.17534102 0.0068755937 -0.09823057 0.5449816 -0.5002402 0.14798155 0.18200895 0.51313376 0.20389211 0.27067447 -0.5195517 -0.41846183 -0.31713825 0.5130303 -0.1523358 0.26005828 -0.03143768 0.16590965 0.16006292 0.19024351 -0.20617065 0.04587804 0.18825795 -0.4398467 -0.3356671 0.3331876 0.37784144 0.12870531 -0.44732788 -0.2212613 0.29156667 -0.13473336 -0.4193956 -0.46390483 0.119228706 -0.044601157 -0.36051777 0.49663186 0.47536448 0.13484167 -0.3709423 -0.3698848 -0.35269707 -0.09707577 0.07946141 0.18278788 0.014521136 -0.29001462 0.2954828 0.29466343 -0.512288; -0.49179217 -0.18828587 0.35010442 -0.40488312 -0.37242943 -0.24305788 0.13253851 -0.24438533 0.33337098 -0.13651232 -0.14140683 0.3138517 -0.519639 -0.24612471 -0.16414389 0.028970625 -0.46775848 0.16975561 -0.31512055 0.033318087 0.21952006 -0.25098774 0.021218138 0.35511452 -0.5787109 -0.33341283 -0.3095253 0.35704195 -0.026469141 -0.32711172 0.36587837 0.20736314 -0.44878003 0.070127934 -0.25566483 0.0057543195 -0.040829044 0.33824798 0.25131726 0.27211544 0.2585004 0.14784993 -0.21598668 0.07248167 -0.05946866 0.37883347 -0.07918622 0.042106874 0.3052406 0.14148341 0.13357 0.07255686 0.4812139 0.11439724 -0.25163868 -0.18054132 0.36027017 0.33751458 -0.2123724 0.08441861 -0.110988684 0.56211936 0.24282438 -0.0003839781 0.077879876 -0.1620174 0.1164611 0.020387348 0.32243907 0.39338222 0.35916618 0.39870217 -0.3854962 0.19371253 -0.009779202 0.27259097 0.24960308 -0.10601768 0.09698446 0.19073582; -0.44331852 -0.3601543 0.47773546 -0.4335764 -0.25583962 0.01935684 0.026476098 -0.24033907 -0.027808147 -0.30562395 0.12304848 0.59842753 -0.40810207 0.055597924 0.0196663 0.079896756 -0.2105244 0.36536923 -0.41508454 -0.05506018 0.22890149 -0.32792598 0.12418265 0.110089496 -0.30858737 -0.21736388 -0.34705073 0.15925965 -0.4353023 -0.36701205 0.22129442 0.15898697 -0.29692233 0.40268067 -0.4486097 0.07552959 -0.06059888 0.0036531254 0.07675434 0.61337644 0.3771212 0.17048451 -0.2858766 0.40092072 -0.042169105 0.28714454 -0.024521068 0.16015393 0.17164716 0.08513999 0.07354126 -0.060301773 0.37115806 0.03861355 -0.31577563 -0.23749635 0.14786841 0.44640693 -0.0189547 -0.1173134 0.0077204187 0.50982416 0.34129936 0.20834735 -0.07969955 -0.10211957 -0.04078845 -0.14819652 0.3541518 0.38570267 0.51219255 0.3609342 -0.38430107 0.23205441 0.02664989 0.15652022 0.21889241 -0.4416029 0.016638193 0.4841746; -0.32171115 -0.106417164 -0.022400714 0.14034715 0.30281648 -0.17160401 0.56808823 -0.09344928 0.15654145 0.31933886 -0.111465946 0.32165357 -0.38209093 -0.3625419 -0.16925026 0.35099033 0.099846974 -0.04763169 -0.39116567 -0.3446773 -0.09188356 -0.18351997 0.5293556 0.14342664 -0.29798383 0.03047756 -0.17309499 0.22856802 0.074096054 -0.109364204 0.12344293 0.49384326 -0.045984287 -0.33438003 0.23403656 0.24602923 0.26975137 0.2673438 0.36431715 0.20063376 -0.14293781 -0.43498883 -0.24757545 -0.20604011 0.6169961 -0.10993515 -0.28315967 0.30190605 0.45801085 0.29038754 0.51702017 0.28716904 0.004435279 -0.54295117 -0.25512743 0.13394721 -0.009177847 0.26854014 -0.2801578 0.26101443 -0.23399006 0.13657187 0.026547465 0.31031573 -0.26472518 -0.5161113 0.06519029 0.23200512 0.007603888 0.09293578 -0.33477485 0.15870962 -0.43390906 0.3017255 0.47766814 0.13303125 -0.26846173 0.011613656 0.31889874 0.2316136; 0.12280781 0.39418483 -0.38653046 0.44282645 0.25849932 -0.17045066 -0.055746276 -0.060034014 -0.046084892 0.36712092 -0.38798 -0.44758013 0.4041341 -0.05491406 -0.08258336 0.24394453 0.13651301 -0.256326 0.53814644 -0.28408852 -0.45424375 0.2715525 -0.024381861 0.20185453 0.38124686 0.21156861 0.23382327 -0.11956134 0.47696853 0.20061986 -0.099404864 -0.1618066 0.5962717 -0.3097644 0.19123834 -0.064479314 0.38491207 0.01657567 0.1193222 -0.5624905 -0.30449963 -0.34945878 0.56831056 -0.105034046 0.14490114 -0.27654976 0.098123506 0.073416434 -0.08728939 -0.2972869 0.10025853 0.08844737 -0.44962943 -0.19270763 0.08079233 0.4280577 -0.108262464 -0.28734118 -0.05254704 0.05529049 -0.0905222 -0.3016498 -0.3795867 0.30312526 -0.05855441 -0.06701467 0.24575849 0.37904784 -0.30914333 -0.27518982 -0.26426023 -0.42004004 0.08261388 -0.0004009104 0.29013303 -0.2379494 -0.3367018 0.25212076 0.17671183 -0.4573951; -0.14467923 0.14561528 -0.465259 0.2531684 0.33767644 -0.46761486 0.3282352 -0.0029771195 0.2536681 0.3832334 -0.52562785 -0.3156206 0.22671962 -0.42757887 -0.45354578 0.441778 0.19630699 -0.27013838 0.09364484 -0.45821145 -0.39481795 0.10168078 0.3817314 0.43021953 0.05476367 0.20116344 0.17482637 0.24179703 0.45998493 0.015236285 0.04917003 0.38095286 0.17876609 -0.18683392 0.19313936 0.42804655 0.46961412 0.52903384 0.24416031 -0.30720827 -0.38542125 -0.117471375 0.30049494 -0.31255654 0.36631972 -0.24761814 -0.21506406 0.024480296 0.2513346 0.12262861 0.41318497 0.1228063 -0.18528661 -0.13900648 0.1625927 0.40613732 0.079646185 0.053611316 -0.45239106 0.2660881 -0.40319455 -0.27577594 -0.3224475 0.51523775 -0.09946087 -0.21211933 0.4586285 0.43847692 -0.12844063 -0.20158693 -0.18974139 -0.4215154 -0.2784023 0.09801322 0.39723524 0.008793168 -0.18956567 0.1505114 0.24175654 -0.5183288; 0.13245168 0.0742878 -0.40621248 0.3636071 0.13615337 -0.30811822 0.025697969 0.32991463 0.08148482 0.3148542 -0.41195112 -0.5620357 0.22259492 -0.2797928 0.002633927 -0.18952009 0.21943961 -0.41210032 0.391581 -0.20323081 -0.41682684 0.20621549 -0.17373368 -0.20598397 0.4965846 0.37830263 0.37954703 -0.3444837 0.45944402 0.40077037 -0.039850783 -0.19484359 0.44403407 -0.47397175 0.31517372 -0.039069675 0.13188492 -0.18056035 -0.07376442 -0.40018404 -0.27007467 -0.3769548 0.40709472 -0.23218955 0.033696625 -0.14990272 0.10356732 0.11345909 -0.072097205 -0.24621588 0.02608691 -0.03882964 -0.4559113 -0.21554701 0.37337872 0.6257928 0.1673306 -0.30931282 -0.0031296802 0.1568731 0.046751108 -0.4012464 -0.19161057 0.16199948 0.07643164 0.0932928 0.2984553 0.30169308 -0.03885996 -0.5760087 -0.21132827 -0.26234376 0.16103458 -0.30247694 0.18011685 -0.05711017 -0.53581005 0.38683894 0.18870674 -0.61487466; -0.30332413 -0.09765809 0.5536596 -0.25969478 -0.17710637 0.13292903 0.36469933 -0.23661128 -0.036863886 -0.45074016 0.03448633 0.28807786 -0.39032704 -0.22383526 0.1288712 0.113222666 -0.27983654 0.51111895 -0.29317155 0.05106021 0.47477272 -0.24995038 0.15815657 0.20554264 -0.22067347 -0.4177385 -0.2601032 0.32117078 -0.3365985 -0.12201252 0.27457666 0.10479062 -0.42519268 0.48436332 -0.40937728 0.03377233 0.1069869 0.18101479 0.0548333 0.33405197 0.3389375 0.20321257 -0.45792305 0.09237422 0.054626774 0.16520622 -0.17750686 0.09696558 0.2353203 0.0916404 0.34906888 0.04934036 0.08761556 -0.16195497 -0.36126786 -0.38348 0.032279585 0.3161455 0.014052255 0.18193772 0.0715612 0.43694627 0.43943512 -0.023258118 -0.2816291 0.08261402 -0.36615714 0.02886203 0.05322511 0.5385491 0.37658742 0.545972 -0.20463736 0.12831178 -0.08833976 -0.17797759 0.6242599 -0.43688825 0.23707645 0.6042837; -0.24526289 -0.27193078 0.4686567 -0.39863807 -0.38863048 0.18369545 0.040057063 -0.3656755 0.07790757 -0.3905257 0.1845772 0.42164326 -0.36290082 0.006430492 0.03201249 0.1456489 -0.22703522 0.19924162 -0.40907392 0.060736228 0.31500742 -0.31717062 -0.10358375 0.06451129 -0.5455054 -0.3796206 -0.3932219 0.29546818 -0.44962433 -0.23139814 -0.03973598 0.15911146 -0.44687304 0.38062218 -0.41870263 0.020045359 -0.28846094 -0.0043901796 0.026963282 0.39557004 0.36582246 0.17082286 -0.6019602 0.14630833 -0.08861996 0.44384888 -0.16790947 -0.07841002 -0.041997727 0.08568078 0.271272 -0.075097695 0.3537214 0.048118737 -0.29863128 -0.48582652 -0.053672902 0.113357976 -0.22605768 -0.2837908 0.04309148 0.41172996 0.25012475 -0.17102103 -0.028994577 -0.0595139 -0.46165663 -0.24824232 0.17670767 0.51529276 0.3255087 0.40242448 -0.36754522 0.026979936 -0.20611316 0.21346067 0.37238127 -0.25270122 -0.15943308 0.66740286; -0.14951904 -0.20908634 0.2089761 -0.051702604 -0.16353223 -0.24620035 0.10563394 -0.05366431 0.30912113 -0.07354307 0.082265764 0.34062788 -0.42485046 -0.1949391 -0.1614228 0.16036318 -0.32441962 0.18401077 -0.49132892 -0.056715313 -0.046846416 -0.43271002 0.25954756 0.13639595 -0.72039324 -0.4489333 -0.2666304 0.24724893 0.0050859977 -0.4244259 0.037403904 0.29568756 -0.22635977 0.19542223 0.014404358 0.10868597 0.10903524 0.03409255 0.30537534 0.52094215 0.15022126 0.1856651 -0.36835703 0.24978341 0.2505714 0.23343623 -0.07613309 0.13015246 0.27363667 0.29085004 0.3087218 -0.012907274 0.28360367 -0.1174901 -0.40191054 -0.075704716 0.04647617 0.37567705 -0.062821046 0.2195001 -0.15221073 0.30467215 0.2665345 0.07979149 0.21110001 -0.10299078 -0.022880474 0.0148003455 0.2844238 0.29482523 0.012095819 0.32120624 -0.14038652 0.028053636 0.25232565 0.042305768 0.011736663 -0.4026612 0.22337128 0.45385867; -0.19405234 -0.22594844 0.2115495 -0.35722876 -0.53074867 0.22179501 0.0008540604 -0.28502217 -0.4063411 -0.39423686 0.5437821 0.5659688 0.016572807 0.1138164 0.2314023 -0.30498368 -0.33990186 0.28064573 -0.40707812 0.15236853 0.43972433 -0.13657433 -0.23223472 -0.043451607 -0.40156588 -0.5919947 -0.21389247 0.08975702 -0.5613461 -0.024061209 -0.09107399 0.04930415 -0.6340711 0.646463 -0.39390597 0.019283408 -0.587826 -0.17646988 -0.25648105 0.5089006 0.5552516 0.31796366 -0.5486169 0.25158912 -0.2095953 0.17065008 -0.19962533 -0.1481113 0.0014563894 0.28525057 -0.083494365 -0.20334579 0.40233994 0.2671476 -0.38702795 -0.51012576 0.14388731 0.3643816 0.121653125 -0.10505566 0.31901243 0.3299261 0.3878134 -0.4702187 -0.14108916 0.33290046 -0.47535115 -0.24619345 0.17890105 0.39128506 0.5630005 0.37736058 -0.22914371 -0.019468889 -0.33838823 -0.034252524 0.5844951 -0.5152633 -0.088350184 0.665252], bias = Float32[-0.13032989; -0.28966972; -0.13756396; 0.042905495; -0.2961647; 0.086586766; -0.04411684; -0.11181793; 0.26724735; 0.103950284; -0.13719517; -0.14018707; 0.20725836; 0.04285382; -0.23462379; 0.11201859; 0.20816502; 0.38888162; -0.19085792; 0.11979479; 0.19448873; 0.24682643; 0.07950181; -0.2889053; -0.13116689; 0.031155588; 0.36888546; 0.35584345; 0.17849843; 0.38685697; -0.2893773; 0.17915061; 0.09460933; 0.15984862; 0.021685963; 0.2586123; -0.2621961; 0.05805723; 0.29853734; -0.18962522; -0.091645956; 0.12760848; 0.22631046; 0.09760226; -0.30397397; 0.05741793; -0.088863604; -0.22189584; 0.003959421; 0.1468025; -0.081784114; -0.03614727; -0.20427255; -0.061445966; 0.14029373; -0.21768562; -0.013150413; -0.045529637; 0.2579944; 0.17268422; -0.08442857; -0.0060593695; 0.0014062888; -0.04283831; 0.027255254; 0.28186622; 0.10380893; 0.2758429; 0.001723795; -0.1292029; 0.10927276; -0.09983487; -0.22837266; -0.16139157; -0.2593886; 0.0028863936; -0.086842746; 0.010456207; 0.17868413; 0.12219586]), layer_3 = (weight = Float32[-0.20669049 0.6139451 0.19422026 0.28296998 0.42926857 -0.11317396 0.11344119 -0.33748168 -0.83589154 -0.7639057 -0.083844095 0.02030775 -1.1061504 -0.268204 0.5191573 0.15180029 0.38605356 -0.9318532 0.07715632 -0.056515712 -1.1844798 -0.14571482 0.17091021 0.10851514 1.3092034 -0.29175812 -0.8353329 -0.6163581 -0.34267092 -0.074865334 0.9657653 -0.1487772 -0.58025014 1.4229808 0.14522451 0.2591271 1.0544556 -0.6003693 -0.8513626 0.4449282 0.042215664 -0.11929594 -0.22461429 -0.30143493 0.943068 0.11037512 0.5584568 0.56737375 0.008385586 -0.42519802 0.5077476 1.1765869 0.08818101 0.61055654 0.19813095 0.16919422 -0.027571391 -0.0858394 -1.0402632 -0.20798361 -1.2608188 0.14067325 -1.2507223 -0.021062136 -0.40508574 0.082961634 0.024548326 -0.9400145 0.64147127 0.29212657 0.13383737 0.017354544 0.9968728 0.019860826 0.7185538 -0.013776711 0.22369657 0.06047803 0.49869323 -0.21775399; 0.022288201 0.98152804 0.058686595 0.17289914 0.44413337 -0.10542478 0.13963704 -0.16798645 -0.75512326 -0.6436645 0.07119554 0.094543874 -0.83688384 -0.42108276 0.47059795 0.07604921 -0.0052412907 -0.80023456 0.0007198646 -0.029123634 -0.6485455 -0.48704788 -0.040479835 0.46252736 0.98263836 0.021405043 -0.865593 -0.8756025 -0.31573376 -0.044644 0.6744475 -0.12275488 -0.5193442 0.8247248 0.10660428 -0.058147475 0.65276283 -0.5805468 -0.8851607 0.43244395 0.01674106 -0.13761137 -0.31257704 -0.15784955 0.8273041 0.0064854277 0.5231564 0.4786946 -0.15770875 -0.47555056 0.48387262 0.92044383 0.18542098 0.552483 0.056048207 0.29125705 -0.13376936 -0.07208617 -0.8450229 -0.45782846 -0.8883139 0.022538131 -0.9083644 0.006485233 -0.4236204 -0.35608363 0.20120381 -0.8074406 0.55982554 0.5004926 0.018538011 0.03704046 0.7755612 0.15112928 0.5737199 0.07112561 0.22859946 -0.104650065 0.16834657 -0.5666135], bias = Float32[0.067911565; 0.32790208])) 
p = p_trained    
callback = function (p, l, pred; doplot = true)
    println(l)
    # plot current prediction against data
    if doplot
    
        plt1 = scatter(collect(etasteps), true_data[1, :]; label = "\\phi data")
        scatter!(plt1, collect(etasteps), pred[1, :]; label = "\\phi prediction")
        scatter!(plt1, collect(etasteps), true_data[2, :]; label = "\\phi ' data")
        scatter!(plt1, collect(etasteps), pred[2, :]; label = "\\phi ' prediction")
            #plt1 = scatter(sol.t, true_data[3, :]; label = "data")
            #scatter!(plt1, sol.t, pred[3, :]; label = "prediction")
            #plt=plot(plt1, plt2)
            
        display(plot(plt1))
    
            
            
    end
    return false
end




callback(result_neuralode2.u, loss_neuralode(result_neuralode2.u)...; doplot = true)


xlabel!("\\eta (dimensionless radius)")
title!("Trained Neural ODE vs Noisy data")

savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\Whitedwarf_dataNeuralODEtrainedvstraining_data20points.png")


#---------------------Forecasting-----------------------#
#------------------------------------------------------

#------------------------------------------------------

open("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Trained_parameters\\p_minimized_moderatenoise.txt","w") do f

    write(f, string(result_neuralode2.minimizer))
end


function dudt_node(u,p,t)
    phi, phiderivative = u
   
    output, _ = dudt2([phi,phiderivative],p,st)
    dphi, dphiderivative = output[1],output[2]
    return [dphi,dphiderivative]
end

#Initial Conditions
  #Psi(0)=1, Psi'(0)=1
etaspan2 = (0.05, 5.325)

#radius range
datasize= 100
etasteps2 = range(etaspan2[1], etaspan2[2]; length = datasize)



#Neural ODE prediction forecasting 
prob_node_extrapolate = ODEProblem(dudt_node,I, etaspan2, p_trained)
_sol_node = solve(prob_node_extrapolate, Tsit5(),saveat = collect(etasteps2))
#Neural ODE Extrapolation scatter plot
p_neuralode = scatter(_sol_node, legend = :topright,markeralpha=0.5, label=["NeuralODE \\phi" "NeuralODE \\phi'"], title="Neural ODE Extrapolation")

#Trained (predicted) DATA up to the 20 elements with the Neural ODE.
p=result_neuralode2.minimizer
prob_neuralode = NeuralODE(dudt2,etaspan; saveat = etasteps)
prediction=(prob_neuralode(I, p, st)[1])

#Plot
scatter!(collect(etasteps), prediction[1, :],color=:black,markershape=:hline; label = "\\phi prediction")
xlabel!("\\eta")
scatter!(collect(etasteps), prediction[2, :],color=:black,markershape=:cross; label = "\\phi ' prediction")

#I need to correct the following line 
#savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\ModerateNoise\\WhitedwarfNODEpredictionvspredictedtrainingdata.png")

#It is corrected 
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\WhitedwarfNODEpredictionvspredictedtrainingdata.png")


#Ground truth full data vs Neural ODE full prediction

p_neuralode = scatter(_sol_node, legend = :topright,markeralpha=0.5, label=["NeuralODE \\phi" "NeuralODE \\phi'"], title="Neural ODE Extrapolation")

scatter!(sol.t, transpose(x1_noise), markershape=:cross,linewidth = 1, xaxis = "\\eta",
     label = ["Noisy \\phi" "Noisy \\phi'"])


#
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\Whitedwarf_NODE_forecastedvsGroundTruthDataNoisy_ODE.png")


#ODE data (NoNoise) vs Neural ODE
p_neuralode = scatter(_sol_node, legend = :topright,markeralpha=0.5, label=["NeuralODE \\phi" "NeuralODE \\phi'"], title="Neural ODE Extrapolation")
scatter!(sol, linewidth = 1,markershape=:cross, xaxis = "\\eta",
     label = ["ODE \\phi " "ODE \\phi'"])


savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\Whitedwarf_NODE_forecastedvsGroundTruthDataNoNoise_ODE.png")     


#Final plot for the results- better formated
scatter(sol.t[1:end-80],Array(x1_noise[:,1:end-80])[1,:],markershape=:cross,color=:blue, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi ", title="White Dwarf model")

scatter!(sol.t[1:end-80],Array(x1_noise[:,1:end-80])[2,:],color=:blue,markershape=:cross, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi'")
xlabel!("\\eta (dimensionless radius)")

#Trained Phi NODE
scatter!(collect(etasteps), prediction[1, :],color=:blue,markeralpha=0.3; label = "Predicted \\phi")

scatter!(collect(etasteps), prediction[2, :],color=:blue, markeralpha=0.3;label = "Predicted \\phi'")
scatter!(sol.t[end-79:end],_sol_node[1,end-79:end],color=:orange,markeralpha=0.6,legend=:right,label="Forecasted \\phi")

scatter!(sol.t[end-79:end],_sol_node[2, end-79:end],color=:orange,markeralpha=0.6,label="Forecasted \\phi'")
title!("Trained Neural ODE")
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\Whitedwarf_forecasted_model.png")

#Final atual version for preprint 



scatter(sol.t[1:end-80],Array(x1_noise[:,1:end-80])[1,:],color=:blue,markeralpha=0.3, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi ", title="White Dwarf model")


scatter!(sol.t[end-79:end],Array(x1_noise[:,21:end])[1,:], color=:red,markeralpha=0.3, label = "Testing \\phi")

plot!(sol.t[1:end-80],prediction[1,:],color=:blue,markeralpha=0.3; label = "Predicted \\phi")
xlabel!("\\eta (dimensionless radius)")

plot!(sol.t[end-80:end],_sol_node[1,end-80:end],color=:red,markeralpha=0.30,label="Forecasted \\phi")
title!("Trained Neural ODE")
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0_20points\\Neural ODE\\Results\\ModerateNoise\\NeuralODEModel_finalversion.png")





